//
//  AppDelegate.swift
//  RFID
//
//  Created by Phidgets on 2018-02-16.
//  Copyright © 2018 Phidgets. All rights reserved.
//

import Cocoa
import Phidget22Swift

@NSApplicationMain
class AppDelegate: NSObject, NSApplicationDelegate {

    @IBOutlet weak var window: NSWindow!
    let ch = RFID()
    @IBOutlet weak var phidgetInfoBox: PhidgetInfoBox!
    
    //GUI
    @IBOutlet weak var tagReadLabel: NSTextField!
    @IBOutlet weak var tagReadProtocolLabel: NSTextField!
    @IBOutlet weak var antennaEnabledCheck: NSButton!
    @IBOutlet weak var tagWriteLabel: NSTextField!
    @IBOutlet weak var protocolPopup: NSPopUpButton!
    @IBOutlet weak var lockCheck: NSButton!
    @IBOutlet weak var tagReadBox: NSBox!
    @IBOutlet weak var tagWriteBox: NSBox!
    
    //Error Window
    @IBOutlet weak var errorEventWindow: NSPanel!
    @IBOutlet var errorEventLog: NSTextView!
    @IBOutlet weak var errorEventCount: NSTextField!
    var errorCounter = 0

    //MARK: View events
    func applicationDidFinishLaunching(_ aNotification: Notification) {
        do {
            //Adjusting window height
            var frame = self.window.frame
            frame.size.height = self.window.minSize.height
            self.window.setFrame(frame, display: true, animate: false)
            self.window.center()
            
            let _ = ch.attach.addHandler(attach_handler)
            let _ = ch.detach.addHandler(detach_handler)
            let _ = ch.error.addHandler(error_handler)
            let _ = ch.tag.addHandler(tagread_handler)
            let _ = ch.tagLost.addHandler(taglost_handler)
            
            /*
             * Please review the Phidget22 channel matching documentation for details on the device
             * and class architecture of Phidget22, and how channels are matched to device features.
             */
            
            /*
             * Specifies the serial number of the device to attach to.
             * For VINT devices, this is the hub serial number.
             *
             * The default is any device.
             */
            // try ch.setDeviceSerialNumber(<YOUR DEVICE SERIAL NUMBER>)
            
            /*
             * For VINT devices, this specifies the port the VINT device must be plugged into.
             *
             * The default is any port.
             */
            // try ch.setHubPort(0)
            
            /*
             * Specifies which channel to attach to.  It is important that the channel of
             * the device is the same class as the channel that is being opened.
             *
             * The default is any channel.
             */
            // try ch.setChannel(0)
            
            /*
             * In order to attach to a network Phidget, the program must connect to a Phidget22 Network Server.
             * In a normal environment this can be done automatically by enabling server discovery, which
             * will cause the client to discovery and connect to available servers.
             *
             * To force the channel to only match a network Phidget, set remote to 1.
             */
            
            // try Net.enableServerDiscovery(serverType: .deviceRemote)
            // try ch.setIsRemote(true)
            
            /*
             openCmdLine allows command line arguments to be passed to this program in order to
             configure channel matching.
             If you have modified the code and added your own channel matching, comment out
             the openCmdLine call, and simply replace it with the following line:
             */
            //try ch.open()
            
            try phidgetInfoBox.openCmdLine(phid: ch as Phidget)
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    func applicationShouldTerminateAfterLastWindowClosed(_ sender: NSApplication) -> Bool {
        return true
    }
    
    func applicationWillTerminate(_ aNotification: Notification) {
        //Safely close example
        do{
            print("Closing")
            ch.attach.removeAllHandlers()
            ch.detach.removeAllHandlers()
            ch.error.removeAllHandlers()
            ch.tag.removeAllHandlers()
            ch.tagLost.removeAllHandlers()
            try ch.close()
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    //MARK: Event handlers
    func attach_handler(sender: Phidget){
        do{
            let attachedDevice = sender as! RFID
            let deviceID = try attachedDevice.getDeviceID()
            let antennaEnabled = try attachedDevice.getAntennaEnabled()
            
            DispatchQueue.main.async {
                self.phidgetInfoBox.fillPhidgetInfo(device: sender)
                self.tagReadBox.isHidden = false
                self.antennaEnabledCheck.state = antennaEnabled ? NSControl.StateValue.on : NSControl.StateValue.off
                if(deviceID == DeviceID.PN_1024){
                    self.tagWriteBox.isHidden = false
                    self.protocolPopup.addItem(withTitle: "EM4100")
                    self.protocolPopup.addItem(withTitle: "ISO11785 FDX B")
                    self.protocolPopup.addItem(withTitle: "PhidgetTAG")
                    self.protocolPopup.selectItem(at: 2)
                }
                
                //Adjusting window height
                var frame = self.window.frame
                frame.size.height = self.window.maxSize.height
                self.window.setFrame(frame, display: true, animate: false)
                self.window.center()
            }
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    func detach_handler(sender: Phidget){
        DispatchQueue.main.async {
            self.phidgetInfoBox.fillPhidgetInfo(device: nil)
            self.tagReadBox.isHidden = true
            self.tagWriteBox.isHidden = true
            
            //Adjusting window height
            var frame = self.window.frame
            frame.size.height = self.window.minSize.height
            self.window.setFrame(frame, display: true, animate: false)
            self.window.center()
        }
    }
    
    func error_handler(sender: Phidget, data: (code: ErrorEventCode, description: String)){
        outputError(errorDescription: data.description, errorCode: data.code.rawValue)
    }
    
    func tagread_handler(sender: RFID, data: (tag: String, proto: RFIDProtocol)){
        DispatchQueue.main.async {
            self.tagReadLabel.stringValue = data.tag
            self.tagReadProtocolLabel.stringValue = (data.proto == RFIDProtocol.phidgetTAG) ? "PhidgetTAG" : (data.proto == RFIDProtocol.EM4100) ? "EM4100" : "ISO11785 FDX B"
        }
    }
    
    func taglost_handler(sender: RFID, data: (tag: String, proto: RFIDProtocol)){
        DispatchQueue.main.async {
            self.tagReadLabel.stringValue = ""
            self.tagReadProtocolLabel.stringValue = ""
        }
    }
    
    //MARK: GUI Controls
    @IBAction func lock(_ sender: Any) {
        let check = sender as! NSButton
        if(check.state == NSControl.StateValue.on){
            let answer = warning(messageText: "Warning!", informativeText: "This action will lock the RFID tag and prevent further writes. \n\nAre you sure you want to continue?")
            if(answer){
                //nothing to do
            }
            else{
                lockCheck.state = NSControl.StateValue.off
            }
        }
    }
    
    @IBAction func writeTag(_ sender: Any) {
        do{
            try ch.write(tagString: tagWriteLabel.stringValue, proto: (protocolPopup.indexOfSelectedItem == 0) ? RFIDProtocol.EM4100 : (protocolPopup.indexOfSelectedItem == 1) ? RFIDProtocol.ISO11785_FDX_B : RFIDProtocol.phidgetTAG, lockTag: (lockCheck.state == NSControl.StateValue.on) ? true : false)
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    @IBAction func enableAntenna(_ sender: Any) {
        do{
            let check = sender as! NSButton
            if(check.state == NSControl.StateValue.on){
                try ch.setAntennaEnabled(true)
            }
            else{
                try ch.setAntennaEnabled(false)
            }
            
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    //MARK: Error
    func outputError(errorDescription:String, errorCode:UInt32){
        DispatchQueue.main.async {
            self.errorCounter += 1
            let outputString = NSAttributedString(string: String(format:"Error %d: %@\n", errorCode, errorDescription))
            self.errorEventLog.textStorage?.beginEditing()
            self.errorEventLog.textStorage?.append(outputString)
            self.errorEventLog.textStorage?.endEditing()
            self.errorEventCount.stringValue = "\(self.errorCounter)"
            if(!self.errorEventWindow.isVisible){
                self.errorEventWindow.setIsVisible(true)
                self.errorEventWindow.center()
            }
        }
        
    }
    
    @IBAction func clearError(_ sender: Any) {
        errorEventLog.textStorage?.setAttributedString(NSAttributedString(string: ""))
        errorCounter = 0
        errorEventCount.stringValue = "0"
    }
    
    func warning(messageText:String, informativeText:String) -> Bool {
        let alert = NSAlert()
        alert.messageText = messageText
        alert.informativeText = informativeText
        alert.alertStyle = .warning
        alert.addButton(withTitle: "OK")
        alert.addButton(withTitle: "Cancel")
        return alert.runModal() == .alertFirstButtonReturn
    }
}

