//
//  PhidgetInfoBox.swift
//  Accelerometer
//
//  Created by Phidgets on 2018-02-12.
//  Copyright © 2018 Phidgets. All rights reserved.
//

import Cocoa
import Phidget22Swift

class PhidgetInfoBox: NSView {
    
    @IBOutlet var view: NSView!
    @IBOutlet var deviceNameLabel: NSTextField!
    @IBOutlet var versionLabel: NSTextField!
    @IBOutlet var channelLabel: NSTextField!
    @IBOutlet var serialNumberText: NSTextField!
    @IBOutlet var serialNumberLabel: NSTextField!
    @IBOutlet var hubPortLabel: NSTextField!
    @IBOutlet var hubPortText: NSTextField!
    @IBOutlet var remoteLabel: NSTextField!
    @IBOutlet var imageView: NSImageView!
    
    override init(frame frameRect: NSRect) {
        super.init(frame: frameRect)
        Bundle.main.loadNibNamed(NSNib.Name.init(rawValue: "PhidgetInfoBox"), owner: self, topLevelObjects: nil)
        view.frame = self.bounds
        self.addSubview(view)
    }
    
    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        Bundle.main.loadNibNamed(NSNib.Name.init(rawValue: "PhidgetInfoBox"), owner: self, topLevelObjects: nil)
        view.frame = self.bounds
        self.addSubview(view)
    }
    
    func fillPhidgetInfo(device:Phidget?){
        do{
        if(device == nil){
            deviceNameLabel.stringValue = "Nothing"
            versionLabel.stringValue = ""
            channelLabel.stringValue = ""
            serialNumberLabel.stringValue = ""
            hubPortLabel.stringValue = ""
            hubPortText.stringValue = ""
            remoteLabel.stringValue = ""
            imageView.image = nil
            return
        }
        
        channelLabel.stringValue = try String(format:"%d",device!.getChannel())
        versionLabel.stringValue = try String(format:"%d",device!.getDeviceVersion())
        deviceNameLabel.stringValue = try device!.getDeviceName()
            
        let deviceClass = try device!.getDeviceClass()
        if(deviceClass == DeviceClass.VINT){
            hubPortText.isHidden = false
            hubPortLabel.isHidden = false
            serialNumberText.stringValue = "Hub Serial Number:"
            serialNumberLabel.stringValue = try String(format:"%d", device!.getDeviceSerialNumber())
            hubPortText.stringValue = "Hub Port:"
            hubPortLabel.stringValue = try String(format:"%d", device!.getHubPort())
        }
        else{
            hubPortText.isHidden = true
            hubPortLabel.isHidden = true
            serialNumberLabel.stringValue = try String(format:"%d", device!.getDeviceSerialNumber())
        }
        
        try setPicture(phid: device!)
        
        remoteLabel.isHidden = try !device!.getIsRemote()
        } catch let err as PhidgetError {
            //do something with error
        } catch {
            print (error)
        }
        
    }
    
    public func openCmdLine(phid:Phidget) throws{
        var serialNumber:Int?, deviceChannel:Int?, hubPort:Int?
        var logFile:String?, serverName:String?, password:String?, label:String?
        var isHubPort:Bool?, remote:Bool?
        
        var i = 0
        let arguments = CommandLine.arguments
        for arg in arguments{
            i += 1
            if(arg == "n"){
                serialNumber = Int(arguments[i])
            }
            else if(arg == "l"){
                logFile = arguments[i]
            }
            else if(arg == "v"){
                hubPort = Int(arguments[i])
            }
            else if(arg == "c"){
                deviceChannel = Int(arguments[i])
            }
            else if(arg == "s"){
                serverName = arguments[i]
            }
            else if(arg == "p"){
                password = arguments[i]
            }
            else if(arg == "L"){
                label = arguments[i]
            }
            else if(arg == "h"){
                isHubPort = true
            }
            else if(arg == "r"){
                remote = true
            }
        }
        
        if(logFile != nil){
            try Log.enable(level: LogLevel.info, destination: logFile)
        }
        
        if(label != nil){
            try phid.setDeviceLabel(label!)
        }
        
        if(serverName != nil){
            try phid.setServerName(serverName!)
            if(password != nil){
                try Net.setServerPassword(serverName: serverName!, password: password!)
            }
        }

        try phid.setChannel(deviceChannel ?? 0)
        try phid.setDeviceSerialNumber(serialNumber ?? Phidget.anySerialNumber)
        try phid.setHubPort(hubPort ?? Phidget.anyHubPort)
        try phid.setIsHubPortDevice(isHubPort ?? false)
        
        if(remote ?? false){
            try phid.setIsRemote(true)
            try Net.enableServerDiscovery(serverType: ServerType.deviceRemote)
        }
        else{
            try phid.setIsLocal(true)
        }
        
        try phid.open()
    }
    
    func setPicture(phid:Phidget) throws{
        var ch = phid
        if(try phid.getIsHubPortDevice()){
            ch = try phid.getHub() ?? phid
        }
        
        var sku = try ch.getDeviceSKU()
        if(sku.range(of: "1018") != nil){
            sku = "1018"
        }
        else{
            sku = sku.components(separatedBy: "/")[0]
        }
        
        let img = NSImage(byReferencingFile: sku + ".png")
        
        imageView.image = nil
        imageView.display()
        if(img?.isValid ?? false){
            let appName = ProcessInfo.processInfo.processName
            if(!(appName == "Phidget Control Panel")){
                NSApp.applicationIconImage = img
            }
            imageView.image = img
            imageView.display()
        }
        
        
    }
    
}
