//
//  AppDelegate.swift
//  DistanceSensor
//
//  Created by Phidgets on 2018-02-13.
//  Copyright © 2018 Phidgets. All rights reserved.
//

import Cocoa
import Phidget22Swift

@NSApplicationMain
class AppDelegate: NSObject, NSApplicationDelegate {

    @IBOutlet weak var window: NSWindow!
    let ch = DistanceSensor()
    @IBOutlet weak var phidgetInfoBox: PhidgetInfoBox!
    
    //GUI
    @IBOutlet weak var changeTriggerSlider: NSSlider!
    @IBOutlet weak var minChangeTriggerLabel: NSTextField!
    @IBOutlet weak var maxChangeTriggerLabel: NSTextField!
    @IBOutlet weak var changeTriggerLabel: NSTextField!
    @IBOutlet weak var dataIntervalSlider: NSSlider!
    @IBOutlet weak var minDataIntervalLabel: NSTextField!
    @IBOutlet weak var maxDataIntervalLabel: NSTextField!
    @IBOutlet weak var dataIntervalLabel: NSTextField!
    @IBOutlet weak var distanceLabel: NSTextField!
    @IBOutlet weak var settingsBox: NSBox!
    @IBOutlet weak var dataBox: NSBox!
    @IBOutlet weak var reflectionsBox: NSBox!
    @IBOutlet weak var reflectionLabelsBox: NSView!
    @IBOutlet weak var quietModeCheck: NSButton!
    @IBOutlet weak var countLabel: NSTextField!
    
    
    
    //Error Window
    @IBOutlet weak var errorEventWindow: NSPanel!
    @IBOutlet var errorEventLog: NSTextView!
    @IBOutlet weak var errorEventCount: NSTextField!
    var errorCounter = 0
    
    //MARK: View events
    func applicationDidFinishLaunching(_ aNotification: Notification) {
        do {
            //Adjusting window height
            var frame = self.window.frame
            frame.size.height = self.window.minSize.height
            self.window.setFrame(frame, display: true, animate: false)
            self.window.center()
            
            let _ = ch.attach.addHandler(attach_handler)
            let _ = ch.detach.addHandler(detach_handler)
            let _ = ch.error.addHandler(error_handler)
            let _ = ch.distanceChange.addHandler(distancechange_handler)
            let _ = ch.sonarReflectionsUpdate.addHandler(reflectionsupdate_handler)
            
            /*
             * Please review the Phidget22 channel matching documentation for details on the device
             * and class architecture of Phidget22, and how channels are matched to device features.
             */
            
            /*
             * Specifies the serial number of the device to attach to.
             * For VINT devices, this is the hub serial number.
             *
             * The default is any device.
             */
            // try ch.setDeviceSerialNumber(<YOUR DEVICE SERIAL NUMBER>)
            
            /*
             * For VINT devices, this specifies the port the VINT device must be plugged into.
             *
             * The default is any port.
             */
            // try ch.setHubPort(0)
            
            /*
             * Specifies which channel to attach to.  It is important that the channel of
             * the device is the same class as the channel that is being opened.
             *
             * The default is any channel.
             */
            // try ch.setChannel(0)
            
            /*
             * In order to attach to a network Phidget, the program must connect to a Phidget22 Network Server.
             * In a normal environment this can be done automatically by enabling server discovery, which
             * will cause the client to discovery and connect to available servers.
             *
             * To force the channel to only match a network Phidget, set remote to 1.
             */
            
            // try Net.enableServerDiscovery(serverType: .deviceRemote)
            // try ch.setIsRemote(true)
            
            /*
             openCmdLine allows command line arguments to be passed to this program in order to
             configure channel matching.
             If you have modified the code and added your own channel matching, comment out
             the openCmdLine call, and simply replace it with the following line:
             */
            //try ch.open()
            
            try phidgetInfoBox.openCmdLine(phid: ch as Phidget)
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    func applicationShouldTerminateAfterLastWindowClosed(_ sender: NSApplication) -> Bool {
        return true
    }
    
    func applicationWillTerminate(_ aNotification: Notification) {
        //Safely close example
        do{
            print("Closing")
            ch.attach.removeAllHandlers()
            ch.detach.removeAllHandlers()
            ch.error.removeAllHandlers()
            ch.distanceChange.removeAllHandlers()
            ch.sonarReflectionsUpdate.removeAllHandlers()
            try ch.close()
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    //MARK: Event handlers
    func attach_handler(sender: Phidget){
        do{
            let attachedDevice = sender as! DistanceSensor
            let minChangeTrigger = try Double(attachedDevice.getMinDistanceChangeTrigger())
            let maxChangeTrigger = try Double(attachedDevice.getMaxDistanceChangeTrigger())
            let changeTrigger = try Double(attachedDevice.getDistanceChangeTrigger())
            let minDataInterval = try attachedDevice.getMinDataInterval()
            let maxDataInterval = try attachedDevice.getMaxDataInterval()
            let dataInterval = try attachedDevice.getDataInterval()
            let deviceID = try attachedDevice.getDeviceID()
            var quietMode:Bool?

            if(deviceID == DeviceID.PN_DST1200){
                quietMode = try attachedDevice.getSonarQuietMode()
            }
            
            DispatchQueue.main.async {
                self.phidgetInfoBox.fillPhidgetInfo(device: sender)
                self.settingsBox.isHidden = false
                self.dataBox.isHidden = false
                if(deviceID == DeviceID.PN_DST1200){
                    self.reflectionsBox.isHidden = false
                }
                
                self.minChangeTriggerLabel.stringValue = String(format:"%d", minChangeTrigger)
                self.maxChangeTriggerLabel.stringValue = String(format:"%d", maxChangeTrigger)
                self.changeTriggerLabel.stringValue = String(format:"%d mm", changeTrigger)
                self.changeTriggerSlider.doubleValue = changeTrigger
                
                self.changeTriggerSlider.minValue = minChangeTrigger
                self.changeTriggerSlider.maxValue = maxChangeTrigger
                self.changeTriggerSlider.doubleValue = changeTrigger
                
                self.minDataIntervalLabel.stringValue = String(format:"%d",minDataInterval)
                self.maxDataIntervalLabel.stringValue = String(format:"%d",maxDataInterval)
                self.dataIntervalLabel.stringValue = String(format:"%d",dataInterval)
                
                self.dataIntervalSlider.minValue = Double(minDataInterval)
                self.dataIntervalSlider.maxValue = Double(maxDataInterval)
                self.dataIntervalSlider.doubleValue = Double(dataInterval)
                self.quietModeCheck.state = quietMode ?? false ? NSControl.StateValue.on : NSControl.StateValue.off
                
                //Adjusting window height
                var frame = self.window.frame
                frame.size.height = self.window.maxSize.height
                if(deviceID != DeviceID.PN_DST1200){
                    frame.size.height -= 175
                }
                self.window.setFrame(frame, display: true, animate: false)
                self.window.center()
            }
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    func detach_handler(sender: Phidget){
        DispatchQueue.main.async {
            self.phidgetInfoBox.fillPhidgetInfo(device: nil)
            self.settingsBox.isHidden = true
            self.dataBox.isHidden = true
            self.reflectionsBox.isHidden = true
            //Adjusting window height
            var frame = self.window.frame
            frame.size.height = self.window.minSize.height
            self.window.setFrame(frame, display: true, animate: false)
            self.window.center()
        }
    }
    
    func error_handler(sender: Phidget, data: (code: ErrorEventCode, description: String)){
        outputError(errorDescription: data.description, errorCode: data.code.rawValue)
    }
    
    func distancechange_handler(sender: DistanceSensor, distance: UInt32){
        DispatchQueue.main.async {
            self.distanceLabel.stringValue = String(format:"%d mm", distance)
        }
    }
    
    func reflectionsupdate_handler(sender: DistanceSensor, data: (distances: [UInt32], amplitudes: [UInt32], count: UInt32)){
        DispatchQueue.main.async {
            self.countLabel.stringValue = "\(data.count)"
            for view in self.reflectionLabelsBox.subviews{
                if(view is NSTextField){
                    let field = view as! NSTextField
                    if(field.tag < 8){
                        if(field.tag < data.count){
                            field.stringValue = "\(data.distances[field.tag])"
                        }
                        else{
                            field.stringValue = ""
                        }
                    }
                    else if(field.tag < 16){
                        if((field.tag - 8) < data.count){
                            field.stringValue = "\(data.amplitudes[field.tag - 8])"
                        }
                        else{
                            field.stringValue = ""
                        }
                    }
                }
            }
        }
    }
    
    //MARK: GUI Controls
    @IBAction func setChangeTrigger(_ sender: Any) {
        let slider = sender as! NSSlider
        do{
            try ch.setDistanceChangeTrigger(UInt32(slider.intValue))
            changeTriggerLabel.stringValue = String(format:"%d mm", slider.intValue)
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setDataInterval(_ sender: Any) {
        let slider = sender as! NSSlider
        do{
            try ch.setDataInterval(UInt32(slider.intValue))
            dataIntervalLabel.stringValue = String(format:"%d ms", UInt32(slider.intValue))
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setQuietMode(_ sender: Any) {
        let check = sender as! NSButton
        do{
            try ch.setSonarQuietMode(check.state == NSControl.StateValue.on ? true : false)
        } catch let err as PhidgetError {
            outputError(errorDescription: err.description, errorCode: err.errorCode.rawValue)
        } catch {
            print (error)
        }
    }
    
    //MARK: Error
    func outputError(errorDescription:String, errorCode:UInt32){
        DispatchQueue.main.async {
            self.errorCounter += 1
            let outputString = NSAttributedString(string: String(format:"Error %d: %@\n", errorCode, errorDescription))
            self.errorEventLog.textStorage?.beginEditing()
            self.errorEventLog.textStorage?.append(outputString)
            self.errorEventLog.textStorage?.endEditing()
            self.errorEventCount.stringValue = "\(self.errorCounter)"
            if(!self.errorEventWindow.isVisible){
                self.errorEventWindow.setIsVisible(true)
                self.errorEventWindow.center()
            }
        }
        
    }
    
    @IBAction func clearError(_ sender: Any) {
        errorEventLog.textStorage?.setAttributedString(NSAttributedString(string: ""))
        errorCounter = 0
        errorEventCount.stringValue = "0"
    }


}

