//
//  ViewController.swift
//  TemperatureSensor
//
//  Created by Phidgets on 2018-01-30.
//  Copyright © 2018 Phidgets. All rights reserved.
//

import UIKit
import Phidget22Swift
class ViewController: UIViewController {

    let ch = TemperatureSensor()
    var phidgetInfoBox:PhidgetInfoBoxViewController! = nil
    
    @IBOutlet var temperatureLabel: UILabel!
    @IBOutlet var dataIntervalSlider: UISlider!
    @IBOutlet var dataIntervalLabel: UILabel!
    @IBOutlet var changeTriggerSlider: UISlider!
    @IBOutlet var changeTriggerLabel: UILabel!
    @IBOutlet var thermocoupleTypeTitle: UILabel!
    @IBOutlet var thermocoupleTypeSegment: UISegmentedControl!
    @IBOutlet var rtdTypeTitle: UILabel!
    @IBOutlet var rtdTypeSegment: UISegmentedControl!
    @IBOutlet var rtdWireTitle: UILabel!
    @IBOutlet var rtdWireSegment: UISegmentedControl!
    @IBOutlet var stackView: UIStackView!
    
    override func viewDidLoad() {
        super.viewDidLoad()
        let font = UIFont.systemFont(ofSize: 10)
        rtdTypeSegment.setTitleTextAttributes([NSAttributedStringKey.font: font],
                                                for: .normal)
        
        do {
            try Net.enableServerDiscovery(serverType: .deviceRemote)
            let _ = ch.attach.addHandler(attach_handler)
            let _ = ch.detach.addHandler(detach_handler)
            let _ = ch.error.addHandler(error_handler)
            let _ = ch.temperatureChange.addHandler(temperaturechange_handler)
            try ch.open()
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    //MARK: Event Handlers
    func attach_handler(sender: Phidget){
        do{
            try phidgetInfoBox.fillPhidgetInfo(phid: sender)
            let attachedDevice = sender as! TemperatureSensor
            
            let subClass = try attachedDevice.getChannelSubclass()
            
            var rtdType:RTDType? = nil
            var rtdWire:RTDWireSetup? = nil
            var thermocoupleType:ThermocoupleType? = nil
            
            let minDataInterval = try Float(attachedDevice.getMinDataInterval())
            let maxDataInterval = try Float(attachedDevice.getMaxDataInterval())
            let dataInterval = try Float(attachedDevice.getDataInterval())
            
            let minTemperatureChangeTrigger = try Float(attachedDevice.getMinTemperatureChangeTrigger())
            let maxTemperatureChangeTrigger = try Float(attachedDevice.getMaxTemperatureChangeTrigger())
            let temperatureChangeTrigger = try Float(attachedDevice.getTemperatureChangeTrigger())
            
            if(subClass == ChannelSubclass.temperatureSensorThermocouple){
                thermocoupleType = try attachedDevice.getThermocoupleType()
            }
            else if (subClass == ChannelSubclass.temperatureSensorRTD){
                rtdType = try attachedDevice.getRTDType()
                rtdWire = try attachedDevice.getRTDWireSetup()
            }
            
            
            DispatchQueue.main.async {
                self.stackView.isHidden = false
                self.dataIntervalSlider.minimumValue = minDataInterval
                self.dataIntervalSlider.maximumValue = maxDataInterval
                self.dataIntervalSlider.value = dataInterval
                self.dataIntervalLabel.text = String(format: "%d ms", UInt32(dataInterval))

                self.changeTriggerSlider.minimumValue = minTemperatureChangeTrigger
                self.changeTriggerSlider.maximumValue = maxTemperatureChangeTrigger
                self.changeTriggerSlider.value = temperatureChangeTrigger
                self.changeTriggerLabel.text = String(format: "%.2f ℃", temperatureChangeTrigger)
                
                if(subClass == ChannelSubclass.temperatureSensorThermocouple){
                    self.updateGUI(isThermocoupleHidden: false, isRTDHidden: true)
                    self.thermocoupleTypeSegment.selectedSegmentIndex = (thermocoupleType == ThermocoupleType.J) ? 0 : (thermocoupleType == ThermocoupleType.K) ? 1 : (thermocoupleType == ThermocoupleType.E) ? 2 : 3
                }
                else if (subClass == ChannelSubclass.temperatureSensorRTD){
                    self.updateGUI(isThermocoupleHidden: true, isRTDHidden: false)
                    self.rtdTypeSegment.selectedSegmentIndex = (rtdType == RTDType.PT100_3850) ? 0 : (rtdType == RTDType.PT1000_3850) ? 1 : (rtdType == RTDType.PT100_3920) ? 2 : 3
                    self.rtdWireSegment.selectedSegmentIndex = (rtdWire == RTDWireSetup.wires_2) ? 0 : (rtdWire == RTDWireSetup.wires_3) ? 1 : 2
                }
                else{
                    self.updateGUI(isThermocoupleHidden: true, isRTDHidden: true)
                }
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
        
    }
    
    func detach_handler(sender: Phidget){
        do{
            try phidgetInfoBox.fillPhidgetInfo(phid: nil)
            DispatchQueue.main.async {
                self.stackView.isHidden = true
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    func error_handler(sender: Phidget, data: (code: ErrorEventCode, description: String)){
        self.outputErrorMessage(errorTitle: "Error Code: \(data.code)", errorMessage: data.description)
    }
    var timeStamp:TimeInterval = 0
    var lastTimeStamp:TimeInterval = 0
    func outputErrorMessage(errorTitle:String, errorMessage:String){
        timeStamp = NSDate.timeIntervalSinceReferenceDate
        //Add a delay for error events so the user does not get bombarded by error events
        if(timeStamp - lastTimeStamp > 5){
            let alert = UIAlertController(title: errorTitle, message: errorMessage, preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "OK", style: .default, handler: {action -> Void in
            }))
            self.present(alert, animated: true, completion: nil)
            lastTimeStamp = timeStamp
        }
    }
    
    func temperaturechange_handler(sender: TemperatureSensor, temperature: Double){
        DispatchQueue.main.async {
            self.temperatureLabel.text = String(format:"%.2f ℃", temperature)
        }
    }
    
    //MARK: GUI controls
    @IBAction func setDataInterval(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setDataInterval(UInt32(slider.value))
            DispatchQueue.main.async {
                self.dataIntervalLabel.text = "\(UInt32(slider.value)) ms"
            }
            
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    @IBAction func setChangeTrigger(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setTemperatureChangeTrigger(Double(slider.value))
            DispatchQueue.main.async {
                self.changeTriggerLabel.text = String(format:"%0.2f ℃", Double(slider.value))
            }
            
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setThermocoupleType(_ sender: Any) {
        let segment = sender as! UISegmentedControl
        do{
            switch(segment.selectedSegmentIndex){
            case 0:
                try ch.setThermocoupleType(ThermocoupleType.J)
                break
            case 1:
                try ch.setThermocoupleType(ThermocoupleType.K)
                break
            case 2:
                try ch.setThermocoupleType(ThermocoupleType.E)
                break
            case 3:
                try ch.setThermocoupleType(ThermocoupleType.T)
                break
            default:
                break
                
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setRTDType(_ sender: Any) {
        let segment = sender as! UISegmentedControl
        do{
            switch(segment.selectedSegmentIndex){
            case 0:
                try ch.setRTDType(RTDType.PT100_3850)
                break
            case 1:
                try ch.setRTDType(RTDType.PT1000_3850)
                break
            case 2:
                try ch.setRTDType(RTDType.PT100_3920)
                break
            case 3:
                try ch.setRTDType(RTDType.PT1000_3920)
                break
            default:
                break
                
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setRTDWireSetup(_ sender: Any) {
        let segment = sender as! UISegmentedControl
        do{
            switch(segment.selectedSegmentIndex){
            case 0:
                try ch.setRTDWireSetup(RTDWireSetup.wires_2)
                break
            case 1:
                try ch.setRTDWireSetup(RTDWireSetup.wires_3)
                break
            case 2:
                try ch.setRTDWireSetup(RTDWireSetup.wires_4)
                break
            default:
                break
                
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    
    
    //MARK: navigation
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "infoBoxSegue" {
            phidgetInfoBox = segue.destination as! PhidgetInfoBoxViewController
        }
    }
    
    //MARK: Misc
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
    }
    
    func updateGUI(isThermocoupleHidden:Bool, isRTDHidden:Bool){
        DispatchQueue.main.async {
            self.thermocoupleTypeTitle.isHidden = isThermocoupleHidden
            self.thermocoupleTypeSegment.isHidden = isThermocoupleHidden
            self.rtdTypeTitle.isHidden = isRTDHidden
            self.rtdTypeSegment.isHidden = isRTDHidden
            self.rtdWireTitle.isHidden = isRTDHidden
            self.rtdWireSegment.isHidden = isRTDHidden
        }
    }
}

