//
//  ViewController.swift
//  Stepper
//
//  Created by Phidgets on 2018-01-31.
//  Copyright © 2018 Phidgets. All rights reserved.
//

import UIKit
import Phidget22Swift

class ViewController: UIViewController {

    let ch = Stepper()
    var phidgetInfoBox:PhidgetInfoBoxViewController! = nil
    
    @IBOutlet var positionLabel: UILabel!
    @IBOutlet var velocityLabel: UILabel!
    @IBOutlet var motorMovingLabel: UILabel!
    @IBOutlet var rescaleLabel: UITextField!
    @IBOutlet var accelerationSlider: UISlider!
    @IBOutlet var accelerationLabel: UILabel!
    @IBOutlet var velocityLimitSlider: UISlider!
    @IBOutlet var velocityLimitLabel: UILabel!
    @IBOutlet var targetPositionSlider: UISlider!
    @IBOutlet var targetPositionLabel: UILabel!
    @IBOutlet var currentLimitTitle: UILabel!
    @IBOutlet var currentLimitSlider: UISlider!
    @IBOutlet var currentLimitLabel: UILabel!
    @IBOutlet var engageButton: UIButton!
    @IBOutlet var stackView: UIStackView!
    
    
    override func viewDidLoad() {
        do {
            try Net.enableServerDiscovery(serverType: .deviceRemote)
            let _ = ch.attach.addHandler(attach_handler)
            let _ = ch.detach.addHandler(detach_handler)
            let _ = ch.error.addHandler(error_handler)
            let _ = ch.positionChange.addHandler(positionchange_handler)
            let _ = ch.velocityChange.addHandler(velocitychange_handler)
            let _ = ch.stopped.addHandler(stopped_handler)
            try ch.open()
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }

    //MARK: Event Handlers
    func attach_handler(sender: Phidget){
        do{
            try phidgetInfoBox.fillPhidgetInfo(phid: sender)
            let attachedDevice = sender as! Stepper
            
            let deviceID = try attachedDevice.getDeviceID()
            var minCurrentLimit:Float = 0, maxCurrentLimit:Float = 0, currentLimit:Float = 0
            let minAcceleration = try Float(attachedDevice.getMinAcceleration())
            let maxAcceleration = try Float(attachedDevice.getMaxAcceleration())
            let acceleration = try Float(attachedDevice.getAcceleration())
            let minVelocityLimit = try Float(attachedDevice.getMinVelocityLimit())
            let maxVelocityLimit = try Float(attachedDevice.getMaxVelocityLimit())
            let velocityLimit = try Float(attachedDevice.getVelocityLimit())
            let minTargetPosition = Float(-10000)
            let maxTargetPosition = Float(10000)
            let targetPosition = try Float(attachedDevice.getTargetPosition())
            let engaged = try attachedDevice.getEngaged()
            
            switch(deviceID){
            case DeviceID.PN_1062:
                self.updateGUI(isCurrentLimitHidden: true)
                break
            default:
                self.updateGUI(isCurrentLimitHidden: false)
                minCurrentLimit = try Float(attachedDevice.getMinCurrentLimit())
                maxCurrentLimit = try Float(attachedDevice.getMaxCurrentLimit())
                currentLimit = try Float(attachedDevice.getCurrentLimit())
                break
            }
            
            DispatchQueue.main.async {
                self.stackView.isHidden = false
                self.accelerationSlider.minimumValue = minAcceleration
                self.accelerationSlider.maximumValue = maxAcceleration
                self.accelerationSlider.value = acceleration
                self.accelerationLabel.text = String(format:"%.3f", acceleration)
                self.velocityLimitSlider.minimumValue = minVelocityLimit
                self.velocityLimitSlider.maximumValue = maxVelocityLimit
                self.velocityLimitSlider.value = velocityLimit
                self.velocityLimitLabel.text = String(format: "%.3f", velocityLimit)
                self.targetPositionSlider.minimumValue = minTargetPosition
                self.targetPositionSlider.maximumValue = maxTargetPosition
                self.targetPositionSlider.value = targetPosition
                self.targetPositionLabel.text = String(format:"%.3f", targetPosition)
                
                if(deviceID != DeviceID.PN_1062){
                    self.currentLimitSlider.minimumValue = minCurrentLimit
                    self.currentLimitSlider.maximumValue = maxCurrentLimit
                    self.currentLimitSlider.value = currentLimit
                    self.currentLimitLabel.text = String(format:"%.2f A", currentLimit)
                }
                
                if(engaged){
                    self.engageButton.setTitle("Disengage", for: UIControlState.normal)
                }
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
        
    }
    
    func detach_handler(sender: Phidget){
        do{
            try phidgetInfoBox.fillPhidgetInfo(phid: nil)
            DispatchQueue.main.async {
                self.stackView.isHidden = true
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    func error_handler(sender: Phidget, data: (code: ErrorEventCode, description: String)){
        self.outputErrorMessage(errorTitle: "Error Code: \(data.code)", errorMessage: data.description)
    }
    var timeStamp:TimeInterval = 0
    var lastTimeStamp:TimeInterval = 0
    func outputErrorMessage(errorTitle:String, errorMessage:String){
        timeStamp = NSDate.timeIntervalSinceReferenceDate
        //Add a delay for error events so the user does not get bombarded by error events
        if(timeStamp - lastTimeStamp > 5){
            let alert = UIAlertController(title: errorTitle, message: errorMessage, preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "OK", style: .default, handler: {action -> Void in
            }))
            self.present(alert, animated: true, completion: nil)
            lastTimeStamp = timeStamp
        }
    }
    
    func positionchange_handler(sender: Stepper, position: Double){
        DispatchQueue.main.async {
            self.positionLabel.text = String(format:"%.3f",position)
            self.updateStopped()
        }
    }
    
    func velocitychange_handler(sender: Stepper, velocity: Double){
        DispatchQueue.main.async {
            self.velocityLabel.text = String(format:"%.3f",velocity)
            self.updateStopped()
        }
    }
    
    func stopped_handler(sender: Stepper){
        DispatchQueue.main.async {
            self.motorMovingLabel.text = "Stopped"
            self.motorMovingLabel.backgroundColor = UIColor.init(red: 1, green: 0.494, blue: 0.474, alpha: 1.0)
        }
    }
    
    //MARK: GUI controls
    @IBAction func setAcceleration(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setAcceleration(Double(slider.value))
            DispatchQueue.main.async {
                self.accelerationLabel.text = String(format:"%0.3f",slider.value)
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setVelocityLimit(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setVelocityLimit(Double(slider.value))
            DispatchQueue.main.async {
                self.velocityLimitLabel.text = String(format:"%0.3f",slider.value)
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setTargetPosition(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setTargetPosition(Double(slider.value))
            DispatchQueue.main.async {
                self.targetPositionLabel.text = String(format:"%0.3f",slider.value)
                self.updateStopped()
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setCurrentLimit(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setCurrentLimit(Double(slider.value))
            DispatchQueue.main.async {
                self.currentLimitLabel.text = String(format:"%0.2f A",slider.value)
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func zeroPosition(_ sender: Any) {
        do{
            let positionOffset = try ch.getPosition()
            try ch.addPositionOffset(positionOffset: -1*positionOffset)
            let currentPosition = try ch.getPosition()
            DispatchQueue.main.async {
                self.positionLabel.text = String(format:"%.3f",currentPosition)
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func engageMotor(_ sender: Any) {
        let button = sender as! UIButton
        do{
            let engaged = try ch.getEngaged()
            try ch.setEngaged(!engaged)
            DispatchQueue.main.async {
                if(engaged){
                    button.setTitle("Engage", for: UIControlState.normal)
                }
                else{
                    button.setTitle("Disengage", for: UIControlState.normal)
                }
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setRescaleFactor(_ sender: Any) {
        do{
            self.view.endEditing(true)
            let rescale = Double(rescaleLabel.text!)
            if(rescale != nil){
                try ch.setRescaleFactor(rescale!)
                let minAcceleration = try Float(ch.getMinAcceleration())
                let maxAcceleration = try Float(ch.getMaxAcceleration())
                let acceleration = try Float(ch.getAcceleration())
                
                let minVelocityLimit = try Float(ch.getMinVelocityLimit())
                let maxVelocityLimit = try Float(ch.getMaxVelocityLimit())
                let velocityLimit = try Float(ch.getVelocityLimit())
                
                let minTargetPosition = Float(-10000 * rescale!)
                let maxTargetPosition = Float(10000 * rescale!)
                let targetPosition = Float(0)
                
                DispatchQueue.main.async {
                    self.accelerationSlider.minimumValue = minAcceleration
                    self.accelerationSlider.maximumValue = maxAcceleration
                    self.accelerationSlider.value = acceleration
                    self.accelerationLabel.text = String(format:"%.3f", acceleration)
                    
                    self.velocityLimitSlider.minimumValue = minVelocityLimit
                    self.velocityLimitSlider.maximumValue = maxVelocityLimit
                    self.velocityLimitSlider.value = velocityLimit
                    self.velocityLimitLabel.text = String(format:"%.3f", velocityLimit)
                    
                    self.targetPositionSlider.minimumValue = minTargetPosition
                    self.targetPositionSlider.maximumValue = maxTargetPosition
                    self.targetPositionSlider.value = targetPosition
                    self.positionLabel.text = String(format:"%.3f",targetPosition)
                }
                
            }
            else{
                outputErrorMessage(errorTitle: "Error", errorMessage: "Rescale factor not valid")
                let currentRescale = try ch.getRescaleFactor()
                DispatchQueue.main.async {
                    self.rescaleLabel.text = "\(currentRescale)"
                }
            }
            
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    //MARK: navigation
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "infoBoxSegue" {
            phidgetInfoBox = segue.destination as! PhidgetInfoBoxViewController
        }
    }
    
    //MARK: Misc
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
    }
    
    func updateGUI(isCurrentLimitHidden:Bool){
        DispatchQueue.main.async {
            self.currentLimitTitle.isHidden = isCurrentLimitHidden
            self.currentLimitSlider.isHidden = isCurrentLimitHidden
            self.currentLimitLabel.isHidden = isCurrentLimitHidden
        }
    }
    
    
    func updateStopped(){
        do{
            let isEngaged = try ch.getEngaged()
            if(!isEngaged){
                return
            }
            let isMoving = try ch.getIsMoving()
            DispatchQueue.main.async {
                if(isMoving){
                    self.motorMovingLabel.text = "Moving"
                    self.motorMovingLabel.backgroundColor = UIColor.init(red: 0.396, green: 0.867, blue: 0.424, alpha: 1.0)
                }
                else{
                    self.motorMovingLabel.text = "Stopped"
                    self.motorMovingLabel.backgroundColor = UIColor.init(red: 1, green: 0.494, blue: 0.474, alpha: 1.0)
                }
            }
            
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
}

