//
//  ViewController.swift
//  CurrentInput
//
//  Created by Phidgets on 2018-01-26.
//  Copyright © 2018 Phidgets. All rights reserved.
//

import UIKit
import Phidget22Swift

class ViewController: UIViewController {

    let ch = CurrentInput()
    var phidgetInfoBox:PhidgetInfoBoxViewController! = nil
    
    @IBOutlet var currentLabel: UILabel!
    @IBOutlet var dataIntervalSlider: UISlider!
    @IBOutlet var dataIntervalLabel: UILabel!
    @IBOutlet var changeTriggerSlider: UISlider!
    @IBOutlet var changeTriggerLabel: UILabel!
    @IBOutlet var stackView: UIStackView!
    @IBOutlet var powerSupplyTitle: UILabel!
    @IBOutlet var powerSupplySegmentedControl: UISegmentedControl!
    
    override func viewDidLoad() {
        super.viewDidLoad()
        do {
            try Net.enableServerDiscovery(serverType: .deviceRemote)
            let _ = ch.attach.addHandler(attach_handler)
            let _ = ch.detach.addHandler(detach_handler)
            let _ = ch.error.addHandler(error_handler)
            let _ = ch.currentChange.addHandler(currentchange_handler)
            try ch.open()
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    //MARK: Event Handlers
    func attach_handler(sender: Phidget){
        do{
            try phidgetInfoBox.fillPhidgetInfo(phid: sender)
            let attachedDevice = sender as! CurrentInput
            let deviceID = try attachedDevice.getDeviceID()
            
            var powerSupply:PowerSupply? = nil
            let minDataInterval = try Float(attachedDevice.getMinDataInterval())
            let maxDataInterval = try Float(attachedDevice.getMaxDataInterval())
            let dataInterval = try Float(attachedDevice.getDataInterval())
            let minCurrentChangeTrigger = try Float(attachedDevice.getMinCurrentChangeTrigger())
            let maxCurrentChangeTrigger = try Float(attachedDevice.getMaxCurrentChangeTrigger())
            let currentChangeTrigger = try Float(attachedDevice.getCurrentChangeTrigger())
            if(deviceID == DeviceID.PN_DAQ1400){
                powerSupply = try attachedDevice.getPowerSupply()
            }
            
            
            DispatchQueue.main.async {
                self.stackView.isHidden = false
                self.dataIntervalSlider.minimumValue = minDataInterval
                self.dataIntervalSlider.maximumValue = maxDataInterval
                self.dataIntervalSlider.value = dataInterval
                self.dataIntervalLabel.text = String(format: "%d ms", UInt32(dataInterval))
                //if min=max interval can not be configured
                if(minDataInterval == maxDataInterval){
                    self.dataIntervalSlider.isEnabled = false
                }
                self.changeTriggerSlider.minimumValue = minCurrentChangeTrigger
                self.changeTriggerSlider.maximumValue = maxCurrentChangeTrigger
                self.changeTriggerSlider.value = currentChangeTrigger
                self.changeTriggerLabel.text = String(format: "%.3f A", currentChangeTrigger)
                
                if(deviceID == DeviceID.PN_DAQ1400){
                    self.powerSupplyTitle.isHidden = false
                    self.powerSupplySegmentedControl.isHidden = false
                    self.powerSupplySegmentedControl.selectedSegmentIndex = (powerSupply == PowerSupply.off) ? 0 : (powerSupply == PowerSupply.volts_12) ? 1 : 2
                }
                else{
                    self.powerSupplyTitle.isHidden = true
                    self.powerSupplySegmentedControl.isHidden = true
                }
            }
            
            
            
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
        
    }
    
    func detach_handler(sender: Phidget){
        do{
            try phidgetInfoBox.fillPhidgetInfo(phid: nil)
            DispatchQueue.main.async {
                self.stackView.isHidden = true
            }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    func error_handler(sender: Phidget, data: (code: ErrorEventCode, description: String)){
        self.outputErrorMessage(errorTitle: "Error Code: \(data.code)", errorMessage: data.description)
    }
    var timeStamp:TimeInterval = 0
    var lastTimeStamp:TimeInterval = 0
    func outputErrorMessage(errorTitle:String, errorMessage:String){
        timeStamp = NSDate.timeIntervalSinceReferenceDate
        //Add a delay for error events so the user does not get bombarded by error events
        if(timeStamp - lastTimeStamp > 5){
            let alert = UIAlertController(title: errorTitle, message: errorMessage, preferredStyle: .alert)
            alert.addAction(UIAlertAction(title: "OK", style: .default, handler: {action -> Void in
            }))
            self.present(alert, animated: true, completion: nil)
            lastTimeStamp = timeStamp
        }
    }
    
    func currentchange_handler(sender: CurrentInput, current: Double){
        DispatchQueue.main.async {
            self.currentLabel.text = String(format:"%.3f A", current)
        }
    }
    
    //MARK: GUI controls
    @IBAction func setDataInterval(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setDataInterval(UInt32(slider.value))
            DispatchQueue.main.async {
                self.dataIntervalLabel.text = "\(UInt32(slider.value)) ms"
            }
            
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    @IBAction func setChangeTrigger(_ sender: Any) {
        let slider = sender as! UISlider
        do{
            try ch.setCurrentChangeTrigger(Double(slider.value))
            DispatchQueue.main.async {
                self.changeTriggerLabel.text = String(format:"%0.3f A", Double(slider.value))
            }
            
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
    }
    
    @IBAction func setPowerSupply(_ sender: Any) {
        let segment = sender as! UISegmentedControl
        
        do{
        switch(segment.selectedSegmentIndex){
        case 0:
            try ch.setPowerSupply(PowerSupply.off)
            break
        case 1:
            try ch.setPowerSupply(PowerSupply.volts_12)
            break
        case 2:
            try ch.setPowerSupply(PowerSupply.volts_24)
            break
        default:
            break
        }
        } catch let err as PhidgetError {
            outputErrorMessage(errorTitle: "Error Code: \(err.errorCode)", errorMessage: err.description)
        } catch {
            print (error)
        }
        
    }
    

    //MARK: navigation
    override func prepare(for segue: UIStoryboardSegue, sender: Any?) {
        if segue.identifier == "infoBoxSegue" {
            phidgetInfoBox = segue.destination as! PhidgetInfoBoxViewController
        }
    }
    
    //MARK: Misc
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
    }
}

