import com.phidget22.*;
import java.util.Scanner; //Required for Text Input

public class IRExample {
	
	static Scanner s = new Scanner(System.in);
	
	public static void main(String[] args) throws Exception {
		
		/***
		* Allocate a new Phidget Channel object
		***/
		IR ch = new IR();

		/**
		* Displays info about the attached Phidget channel.
		* Fired when a Phidget channel with onAttachHandler registered attaches
		*/
		ch.addAttachListener(new AttachListener() {
			public void onAttach(AttachEvent ae) {
			
				try {
					//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
					//www.phidgets.com/docs/Using_Multiple_Phidgets for information

					System.out.print("\nAttach Event:");
					
					IR ph = (IR) ae.getSource();
					
					/**
					* Get device information and display it.
					**/
					int serialNumber = ph.getDeviceSerialNumber();
					String channelClass = ph.getChannelClassName();
					int channel = ph.getChannel();
					
					DeviceClass deviceClass = ph.getDeviceClass();
					if (deviceClass != DeviceClass.VINT) {
						System.out.print("\n\t-> Channel Class: " + channelClass + "\n\t-> Serial Number: " + serialNumber +
							  "\n\t-> Channel:  " + channel + "\n");
					} 
					else {			
						int hubPort = ph.getHubPort();
						System.out.print("\n\t-> Channel Class: " + channelClass + "\n\t-> Serial Number: " + serialNumber +
							  "\n\t-> Hub Port: " + hubPort + "\n\t-> Channel:  " + channel + "\n");
					}
					
				} 
				catch (PhidgetException e) {
					PhidgetHelperFunctions.DisplayError(e, "Getting Channel Informaiton");
				}
			
			}
			
		});

		
		/**
		* Displays info about the detached Phidget channel.
		* Fired when a Phidget channel with onDetachHandler registered detaches
		*/
		ch.addDetachListener(new DetachListener() {
			public void onDetach(DetachEvent de) {
				try {
					//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
					//www.phidgets.com/docs/Using_Multiple_Phidgets for information

					System.out.print("\nAttach Event:");
					
					Phidget ph = de.getSource();
					
					/**
					* Get device information and display it.
					**/
					int serialNumber = ph.getDeviceSerialNumber();
					String channelClass = ph.getChannelClassName();
					int channel = ph.getChannel();
					
					DeviceClass deviceClass = ph.getDeviceClass();
					if (deviceClass != DeviceClass.VINT) {
						System.out.print("\n\t-> Channel Class: " + channelClass + "\n\t-> Serial Number: " + serialNumber +
							  "\n\t-> Channel:  " + channel + "\n");
					} 
					else {			
						int hubPort = ph.getHubPort();
						System.out.print("\n\t-> Channel Class: " + channelClass + "\n\t-> Serial Number: " + serialNumber +
							  "\n\t-> Hub Port: " + hubPort + "\n\t-> Channel:  " + channel + "\n");
					}
				} 
				catch (PhidgetException e) {
					PhidgetHelperFunctions.DisplayError(e, "Getting Channel Informaiton");
				}
			}
		});

		/**
		* Writes Phidget error info to stderr.
		* Fired when a Phidget channel with onErrorHandler registered encounters an error in the library
		*/
		ch.addErrorListener(new ErrorListener() {
			public void onError(ErrorEvent ee) {
				System.out.println("Error: " + ee.getDescription());
			}
		});

		/**
		* Outputs the IR's most recently detected code.
		* Fired when a IR channel with onCodeHandler registered detects a code
		*/
		ch.addCodeListener(new IRCodeListener() {
			public void onCode(IRCodeEvent e) {
				//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
				//www.phidgets.com/docs/Using_Multiple_Phidgets for information

				System.out.println("\n[Code Event] -> Code    : " + e.getCode());
				System.out.println("             -> bitCount: " + e.getBitCount());
				System.out.println("             -> isRepeat: " + e.getIsRepeat());
			}
		});

		
		/**
		* Outputs the IR's most recently learned code.
		* Fired when a IR channel with onCodeHandler registered learns a code
		*/
		ch.addLearnListener(new IRLearnListener() {
			public void onLearn(IRLearnEvent e) {
				//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
				//www.phidgets.com/docs/Using_Multiple_Phidgets for information

				System.out.println("\n[Learn Event] -> Code: " + e.getCode());
				
				String encodingString, lengthString;
				int toggleMaskBytes;

				switch (e.getCodeInfo().encoding) {
					case UNKNOWN:
						encodingString = "Unknown";
						break;
					case SPACE:
						encodingString = "Space";
						break;
					case PULSE:
						encodingString = "Pulse";
						break;
					case BI_PHASE:
						encodingString = "BiPhase";
						break;
					case RC5:
						encodingString = "RC5";
						break;
					case RC6:
						encodingString = "RC6";
						break;
					default:
						encodingString = "Unknown";
						break;
				}

				switch (e.getCodeInfo().length) {
					case UNKNOWN:
						lengthString = "Unknown";
						break;
					case CONSTANT:
						lengthString = "Constant";
						break;
					case VARIABLE:
						lengthString = "Variable";
						break;
					default:
						lengthString = "Unknown";
						break;
				}

				System.out.println("Learned Code Info");
				System.out.println("----------------------------------------------------");
				System.out.println("Bit Count: " + e.getCodeInfo().bitCount + "\nEncoding: " + encodingString + "\nLength: " + lengthString + "\nGap: " + e.getCodeInfo().gap + "\nTrail: " + e.getCodeInfo().trail);
				System.out.println("Header: { " + e.getCodeInfo().header[0] + "," + e.getCodeInfo().header[1] + " }\nZero: {" + e.getCodeInfo().one[0] + "," + e.getCodeInfo().one[1] + " }");
				System.out.print("Repeat: {");
				for (int i = 0; i < 26; i++) {
					if (e.getCodeInfo().repeat[i] == 0) {
						break;
					}
					if (i == 0) {
						System.out.print(e.getCodeInfo().repeat[i]);
					} else {
						System.out.print(" ," + e.getCodeInfo().repeat[i]);
					}
				}
				System.out.println("}\nMinRepeat: " + e.getCodeInfo().minRepeat);
				System.out.println("Toggle Mask: " + e.getCodeInfo().toggleMask);
				
				System.out.println("Carrier Frequency: " + e.getCodeInfo().carrierFrequency + "\nDuty Cycle: " + e.getCodeInfo().dutyCycle + "\n");
				System.out.println("----------------------------------------------------\n");
			}
		});

		
		try {
			
			/***
			* Set matching parameters to specify which channel to open
			***/
		
			//You may remove these lines and hard-code the addressing parameters to fit your application
			ChannelInfo channelInfo = new ChannelInfo();  //Information from AskForDeviceParameters(). May be removed when hard-coding parameters.
			PhidgetHelperFunctions.AskForDeviceParameters(channelInfo, ch);
			
			ch.setDeviceSerialNumber(channelInfo.deviceSerialNumber);
			ch.setHubPort(channelInfo.hubPort);
			ch.setIsHubPortDevice(channelInfo.isHubPortDevice);
			ch.setChannel(channelInfo.channel);
			
			if(channelInfo.netInfo.isRemote) {
				ch.setIsRemote(channelInfo.netInfo.isRemote);
				if(channelInfo.netInfo.serverDiscovery) {
					try {
						Net.enableServerDiscovery(ServerType.DEVICE_REMOTE);
					}
					catch (PhidgetException e) {
						PhidgetHelperFunctions.PrintEnableServerDiscoveryErrorMessage(e);
						throw new Exception("Program Terminated: EnableServerDiscovery Failed", e);
					}
				}
				else {
					Net.addServer("Server", channelInfo.netInfo.hostname,
						channelInfo.netInfo.port, channelInfo.netInfo.password, 0);
				}
			}
			
			
			//This call may be harmlessly removed
			PrintEventDescriptions();
			
			/***
			* Open the channel with a timeout
			***/
			System.out.print("\nOpening and Waiting for Attachment...");
			
			try {
				ch.open(5000);
			}
			catch (PhidgetException e) {
				PhidgetHelperFunctions.PrintOpenErrorMessage(e, ch);
				throw new Exception ("Program Terminated: Open Failed", e);
			}
			
			/***
			* To find additional functionality not included in this example,
			* be sure to check the API for your device.
			***/
			
			System.out.println("Sampling data for 10 seconds...");

			Thread.sleep(10000);

			/***
			* Perform clean up and exit
			***/			
			System.out.println("\nDone Sampling...");

			System.out.println("Cleaning up...");
			ch.close();
			System.out.print("\nExiting...");
			return;

			
		} catch (PhidgetException ex) {
			System.out.println(ex);
		}
	}
	
	/***
	* Prints descriptions of how events related to this class work
	***/
	public static void PrintEventDescriptions()	{
		System.out.print("\n--------------------\n" +
				"\n  | Code events will call their associated function every time new IR code data is received from the device.\n" +
				"  | Press ENTER once you have read this message.\n" +
				"\n  | Learn events will call their associated function every time new IR code data is learned.\n" +
				"  | Press ENTER once you have read this message.\n");
		s.nextLine();
		
		System.out.println("\n--------------------");
	}
}
