#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <phidget22.h>
#include "PhidgetHelperFunctions.h"


/**
* Sets the DataInterval and Engages the Stepper
* Displays info about the attached Phidget channel.
* Fired when a Phidget channel with onAttachHandler registered attaches
*
* @param ph The Phidget channel that fired the attach event
* @param *ctx Context pointer. Used to pass information to the event handler.
*/
static void CCONV onAttachHandler(PhidgetHandle ph, void *ctx) {
	PhidgetReturnCode prc; //Used to catch error codes from each Phidget function call
	Phidget_DeviceClass deviceClass;
	const char* channelClassName;
	int32_t serialNumber;
	int32_t hubPort;
	int32_t channel;

	//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
	//www.phidgets.com/docs/Using_Multiple_Phidgets for information

	printf("\nAttach Event: ");
	/*
	* Get device information and display it.
	*/
	prc = Phidget_getDeviceSerialNumber(ph, &serialNumber);
	CheckError(prc, "Getting DeviceSerialNumber", (PhidgetHandle *)&ph);

	prc = Phidget_getChannel(ph, &channel);
	CheckError(prc, "Getting Channel", (PhidgetHandle *)&ph);

	prc = Phidget_getChannelClassName(ph, &channelClassName);
	CheckError(prc, "Getting Channel Class Name", (PhidgetHandle *)&ph);

	prc = Phidget_getDeviceClass(ph, &deviceClass);
	CheckError(prc, "Getting Device Class", (PhidgetHandle *)&ph);

	if (deviceClass == PHIDCLASS_VINT) {
		prc = Phidget_getHubPort(ph, &hubPort);
		CheckError(prc, "Getting HubPort", (PhidgetHandle *)&ph);

		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Hub Port: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, hubPort, channel);
	} else { //Not VINT
		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, channel);
	}

	/*
	*	Set the DataInterval inside of the attach handler to initialize the device with this value.
	*	DataInterval defines the minimum time between PositionChange events.
	*	DataInterval can be set to any value from MinDataInterval to MaxDataInterval.
	*/
	printf("\tSetting DataInterval to 1000ms\n");
	prc = PhidgetStepper_setDataInterval((PhidgetStepperHandle)ph, 1000);
	CheckError(prc, "Setting DataInterval", (PhidgetHandle *)&ph);

	/*
	* Engage the Stepper inside of the attach handler to allow the motor to move to its target position
	* The motor will only track a target position if it is engaged.
	* Engaged can be set to True to enable the servo, or False to disable it.
	*/
	printf("\tEngaging Stepper\n");
	prc = PhidgetStepper_setEngaged((PhidgetStepperHandle)ph, 1);
	CheckError(prc, "Engaging Stepper", (PhidgetHandle *)&ph);
}

/**
* Displays info about the detached Phidget channel.
* Fired when a Phidget channel with onDetachHandler registered detaches
*
* @param ph The Phidget channel that fired the detach event
* @param *ctx Context pointer. Used to pass information to the event handler.
*/
static void CCONV onDetachHandler(PhidgetHandle ph, void *ctx) {
	PhidgetReturnCode prc; //Used to catch error codes from each Phidget function call
	Phidget_DeviceClass deviceClass;
	const char* channelClassName;
	int32_t serialNumber;
	int32_t hubPort;
	int32_t channel;

	//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
	//www.phidgets.com/docs/Using_Multiple_Phidgets for information

	printf("\nDetach Event: ");
	/*
	* Get device information and display it.
	*/
	prc = Phidget_getDeviceSerialNumber(ph, &serialNumber);
	CheckError(prc, "Getting DeviceSerialNumber", (PhidgetHandle *)&ph);

	prc = Phidget_getChannel(ph, &channel);
	CheckError(prc, "Getting Channel", (PhidgetHandle *)&ph);

	prc = Phidget_getChannelClassName(ph, &channelClassName);
	CheckError(prc, "Getting Channel Class Name", (PhidgetHandle *)&ph);

	prc = Phidget_getDeviceClass(ph, &deviceClass);
	CheckError(prc, "Getting Device Class", (PhidgetHandle *)&ph);

	if (deviceClass == PHIDCLASS_VINT) {
		prc = Phidget_getHubPort(ph, &hubPort);
		CheckError(prc, "Getting HubPort", (PhidgetHandle *)&ph);

		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Hub Port: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, hubPort, channel);
	} else { //Not VINT
		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, channel);
	}
}

/**
* Writes Phidget error info to stderr.
* Fired when a Phidget channel with onErrorHandler registered encounters an error in the library
*
* @param ph The Phidget channel that fired the error event
* @param *ctx Context pointer. Used to pass information to the event handler.
* @param errorCode the code associated with the error of enum type Phidget_ErrorEventCode
* @param *errorString string containing the description of the error fired
*/
static void CCONV onErrorHandler(PhidgetHandle ph, void *ctx, Phidget_ErrorEventCode errorCode, const char *errorString) {

	fprintf(stderr, "[Phidget Error Event] -> %s (%d)\n", errorString, errorCode);
}

/**
* Outputs the Stepper's most recently detected position change
* Fired when a Stepper channel with onPositionChangeHandler registered detects a position change
*
* @param ph The Stepper channel that fired the PositionChange event
* @param *ctx Context pointer. Used to pass information to the event handler.
* @param position The reported position from the Stepper channel
*/
static void CCONV onPositionChangeHandler(PhidgetStepperHandle ph, void *ctx, double position) {

	//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
	//www.phidgets.com/docs/Using_Multiple_Phidgets for information

	printf("[Position Event] -> Position: %lf\n", position);
}


/**
* Prints descriptions of the available events for this class
*/
void PrintEventDescriptions() {

	printf("\n--------------------\n"
		"\n  | Position change events will call their associated function every time new position data is received from the device.\n"
		"  | The rate of these events can be set by adjusting the DataInterval for the channel.\n"
		"  | Press ENTER once you have read this message.\n");
	getchar();

	printf("\n--------------------\n");
}

/**
* Creates, configures, and opens a Stepper channel.
* Sets stepper motor target position based on user input
* Closes out Stepper channel
*
* @return 0 if the program exits successfully, 1 if it exits with errors.
*/
int main() {
	PhidgetStepperHandle ch = NULL;
	ChannelInfo channelInfo; //Information from AskForDeviceParameters(). May be removed when hard-coding parameters.
	PhidgetReturnCode prc; //Used to catch error codes from each Phidget function call

	/*
	* Allocate a new Phidget Channel object
	*/

	prc = PhidgetStepper_create(&ch);
	CheckError(prc, "Creating Channel", (PhidgetHandle *)&ch);

	/*
	* Set addressing parameters to specify which channel to open
	*/

	//You can safely remove this line and hard-code the parameters to make the program can start without user input
	AskForDeviceParameters(&channelInfo, (PhidgetHandle *)&ch);

	prc = Phidget_setDeviceSerialNumber((PhidgetHandle)ch, channelInfo.deviceSerialNumber);
	CheckError(prc, "Setting DeviceSerialNumber", (PhidgetHandle *)&ch);

	prc = Phidget_setHubPort((PhidgetHandle)ch, channelInfo.hubPort);
	CheckError(prc, "Setting HubPort", (PhidgetHandle *)&ch);

	prc = Phidget_setChannel((PhidgetHandle)ch, channelInfo.channel);
	CheckError(prc, "Setting Channel", (PhidgetHandle *)&ch);

	if (channelInfo.netInfo.isRemote) {
		prc = Phidget_setIsRemote((PhidgetHandle)ch, channelInfo.netInfo.isRemote);
		CheckError(prc, "Setting IsRemote", (PhidgetHandle *)&ch);
		if (channelInfo.netInfo.serverDiscovery) {
			prc = PhidgetNet_enableServerDiscovery(PHIDGETSERVER_DEVICEREMOTE);
			CheckEnableServerDiscoveryError(prc, (PhidgetHandle *)&ch);
		} else {
			prc = PhidgetNet_addServer("Server", channelInfo.netInfo.hostname,
				channelInfo.netInfo.port, channelInfo.netInfo.password, 0);
			CheckError(prc, "Adding Server", (PhidgetHandle *)&ch);
		}
	}

	/*
	* Add event handlers before calling open so that no events are missed.
	*/

	printf("\n--------------------------------------\n");
	printf("\nSetting OnAttachHandler...\n");
	prc = Phidget_setOnAttachHandler((PhidgetHandle)ch, onAttachHandler, NULL);
	CheckError(prc, "Setting OnAttachHandler", (PhidgetHandle *)&ch);

	printf("Setting OnDetachHandler...\n");
	prc = Phidget_setOnDetachHandler((PhidgetHandle)ch, onDetachHandler, NULL);
	CheckError(prc, "Setting OnDetachHandler", (PhidgetHandle *)&ch);

	printf("Setting OnErrorHandler...\n");
	prc = Phidget_setOnErrorHandler((PhidgetHandle)ch, onErrorHandler, NULL);
	CheckError(prc, "Setting OnErrorHandler", (PhidgetHandle *)&ch);

	//This call may be harmlessly removed
	PrintEventDescriptions();

	printf("Setting OnPositionChangeHandler...\n");
	prc = PhidgetStepper_setOnPositionChangeHandler(ch, onPositionChangeHandler, NULL);
	CheckError(prc, "Setting OnPositionChangeHandler", (PhidgetHandle *)&ch);

	/*
	* Open the channel with a timeout
	*/
	printf("Opening and Waiting for Attachment...\n");
	prc = Phidget_openWaitForAttachment((PhidgetHandle)ch, 5000);
	CheckOpenError(prc, (PhidgetHandle *)&ch);

	char buf[100];
	char* endptr;
	int end = 0;
	double position = 0.0;
	double minPosition = 0.0;
	double maxPosition = 0.0;

	printf("--------------------\n"
		"\n  | Stepper motor position can be controlled by setting its Target Position.\n"
		"  | The target position can be a number between MinPosition and MaxPosition.\n"
		"  | For this example, acceleration and velocity limit are left as their default values, but can be changed in custom code.\n"

		"\nInput a desired position and press ENTER\n"
		"Input Q and press ENTER to quit\n");

	/*
	* To find additional functionality not included in this example,
	* be sure to check the API for your device.
	*/

	prc = PhidgetStepper_getMinPosition(ch, &minPosition);
	CheckError(prc, "Getting Min Position", (PhidgetHandle *)&ch);

	prc = PhidgetStepper_getMaxPosition(ch, &maxPosition);
	CheckError(prc, "Getting Max Position", (PhidgetHandle *)&ch);

	while (!end) {

		//Get user input
		if (fgets(buf, 100, stdin) == NULL)
			continue;

		//Process user input
		if (buf[0] == 'Q' || buf[0] == 'q') {
			end = 1;
			continue;
		}

		position = strtod(buf, &endptr);
		if ((*endptr != '\0') && (isspace(*endptr) == 0))
			continue;

		if (position > maxPosition || position < minPosition) {
			printf("Position must be between %.2f and %.2f\n", minPosition, maxPosition);
			continue;
		}

		//Send the value to the device
		prc = PhidgetStepper_setTargetPosition(ch, position);
		CheckError(prc, "Setting Position", (PhidgetHandle *)&ch);
	}

	/*
	* Perform clean up and exit
	*/
	printf("\nDone Sampling...\n");

	printf("Cleaning up...\n");
	prc = Phidget_close((PhidgetHandle)ch);
	CheckError(prc, "Closing Channel", (PhidgetHandle *)&ch);
	prc = PhidgetStepper_delete(&ch);
	CheckError(prc, "Deleting Channel", (PhidgetHandle *)&ch);
	printf("\nExiting...\n");
	printf("Press ENTER to end program.\n");
	getchar();

	return 0;

}
