#include <stdio.h>
#include <stdlib.h>

#include <phidget22.h>

#ifndef _WIN32
#include <unistd.h>
#else
#include <Windows.h>
#endif

static void CCONV ssleep(int);

// Demos for the more complex log functions. These can be ignored for most purposes. 
// Call them somewhere in main to run them.

void demo_log_level();

void demo_rotation();

void demo_custom_source();

void demo_source_level();

void demo_list_sources();


int
main(int argc, char **argv) {

	PhidgetReturnCode res = 0;
	
	//Let's start by logging some messages to the console to easily see the results
	printf("\nLogging to the Console\n");

	//First, we enable the log
	PhidgetLog_enable(PHIDGET_LOG_INFO, NULL);

	printf("You should see both log messages.\n");

	//You can send your own messages to the Phidget log
	PhidgetLog_log(PHIDGET_LOG_ERROR, "Custom Log Error %d", 0);

	PhidgetLog_log(PHIDGET_LOG_INFO, "Custom Log Info %d", 1);

	ssleep(3);

	//Now we'll disable the log, so we can set it up to write to a file
	PhidgetLog_disable();
	
	//Next let's log to a file	
	printf("\nLogging to File\n");

	//Here, we'll log to a file called log.txt in the same folder as the C program
	PhidgetLog_enable(PHIDGET_LOG_INFO, "log.txt");
	
	//You can send your own messages to the Phidget log
	PhidgetLog_log(PHIDGET_LOG_ERROR, "Custom Log Error %d", 2);

	PhidgetLog_log(PHIDGET_LOG_INFO, "Custom Log Info %d", 3);
	
	PhidgetLog_disable();

	printf("\nNow you can open the log file to view the results\n");

	ssleep(10);

	exit(res);
}

static void CCONV
ssleep(int tm) {
#ifdef _WIN32
	Sleep(tm * 1000);
#else
	sleep(tm);
#endif
}

// Demo function implementation. Here is source code to demonstrate the more obscure
// functionality of the Log object.

void demo_log_level() {
	//Log levels serve to filter out unwanted information from the log

	//To demonstrate, we'll enable the log with log level PHIDGET_LOG_INFO
	PhidgetLog_enable(PHIDGET_LOG_INFO, "log_level_log.txt");

	printf("\nDemonstrating log level PHIDGET_LOG_INFO\n");
	printf("You should see both log messages in log_level_log.txt\n");

	//You can send your own messages to the Phidget log
	PhidgetLog_log(PHIDGET_LOG_INFO, "Custom Log Info %d", 0);

	PhidgetLog_log(PHIDGET_LOG_ERROR, "Custom Log Error %d", 1);

	//Now we'll repeat the precess, using log level ERROR
	PhidgetLog_setLevel(PHIDGET_LOG_ERROR);

	printf("\nDemonstrating log level PHIDGET_LOG_ERROR\n");
	printf("You should only see the log message produced by PHIDGET_LOG_ERROR.\n");

	//Note that the message flagged as INFO will not appear when using log level ERROR
	//This helps filter out unwanted informaiton from the log
	PhidgetLog_log(PHIDGET_LOG_INFO, "Custom Log Info %d", 2);

	PhidgetLog_log(PHIDGET_LOG_ERROR, "Custom Log Error %d", 3);

	PhidgetLog_disable();

	ssleep(5);
}

void demo_rotation() {
	//We'll set up a rotation to keep the file size to 50kB max, 
	//and allow storage of the latest 3 log files
	PhidgetLog_setRotating(50000, 3);

	//Here, we'll log to a file called rotation_log.txt in the same folder as the C program
	PhidgetLog_enable(PHIDGET_LOG_INFO, "rotation_log.txt");

	printf("\nLogging to rotation_log.txt File...\n");
	//Now we'll log 1000 messages to the log file
	//Once complete, open the file to see the results of logging and rotation
	for (int i = 0; i < 1000; i++) {
		PhidgetLog_log(PHIDGET_LOG_INFO, "Custom Log Sequential Logs: %d", i);
	}

	PhidgetLog_disable();

	printf("\nNow you can open rotation_log.txt to view the results\n");

	ssleep(5);
}

void demo_custom_source() {
	//Here we'll demonstrate the use of custom log sources, which may be useful to add customized logging
	//to your applicaiton.
	PhidgetLog_enable(PHIDGET_LOG_INFO, "custom_source_log.txt");

	printf("\nDemonstrating PhidgetLog_loge\n");

	PhidgetLog_addSource("demo_src", PHIDGET_LOG_INFO);

	PhidgetLog_loge("Logging_Example.c", 113, "demo_loge", "demo_src", PHIDGET_LOG_INFO, "Loge Test %d", 1234);

	printf("\nNow you can open custom_source_log.txt to view the results\n");

	PhidgetLog_disable();
}

void demo_source_level() {
	//Setting up a manager to be used to demonstrate logging
	//Not required, but serves as a convenient source of log entries
	PhidgetManagerHandle manager;
	PhidgetManager_create(&manager);

	printf("\nDemonstrating Source Level\n");

	PhidgetLog_enable(PHIDGET_LOG_INFO, "source_level_log.txt");

	printf("\nSetting source _phidget22channel to PHIDGET_LOG_VERBOSE\n");

	//Knowing the list of sources, you can set a specific source to a different
	//log level than the others, to allow more or less detail for that source
	PhidgetLog_setSourceLevel("_phidget22channel", PHIDGET_LOG_VERBOSE);

	printf("\nOpening Manager: Plug and unplug Phidgets, then check source_level_log.txt to\nsee what happened.\n");

	PhidgetManager_open(manager);

	ssleep(5);

	printf("\nClosing Manager\n");

	PhidgetManager_close(manager);

	ssleep(1);

	PhidgetLog_disable();

	ssleep(10);

}

void demo_list_sources() {
	//Listing Log Sources
	printf("\nListing Log Sources\n");

	char *sources[20] = { NULL };
	int sourcesCnt = 20;
	PhidgetLog_getSources(sources, &sourcesCnt);

	for (int i = 0; i < sourcesCnt; i++)
		printf("%s\n", sources[i]);

	ssleep(5);
}