#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <phidget22.h>
#include "PhidgetHelperFunctions.h"


/**
* Displays info about the attached Phidget channel.
* Fired when a Phidget channel with onAttachHandler registered attaches
*
* @param ph The Phidget channel that fired the attach event
* @param *ctx Context pointer. Used to pass information to the event handler.
*/
static void CCONV onAttachHandler(PhidgetHandle ph, void *ctx) {
	PhidgetReturnCode prc; //Used to catch error codes from each Phidget function call
	Phidget_DeviceClass deviceClass;
	char* channelClassName;
	int32_t serialNumber;
	int32_t hubPort;
	int32_t channel;

	//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
	//www.phidgets.com/docs/Using_Multiple_Phidgets for information

	printf("\nAttach Event: ");
	/*
	* Get device information and display it.
	*/
	prc = Phidget_getDeviceSerialNumber(ph, &serialNumber);
	CheckError(prc, "Getting DeviceSerialNumber", (PhidgetHandle *)&ph);

	prc = Phidget_getChannel(ph, &channel);
	CheckError(prc, "Getting Channel", (PhidgetHandle *)&ph);

	prc = Phidget_getChannelClassName(ph, &channelClassName);
	CheckError(prc, "Getting Channel Class Name", (PhidgetHandle *)&ph);

	prc = Phidget_getDeviceClass(ph, &deviceClass);
	CheckError(prc, "Getting Device Class", (PhidgetHandle *)&ph);

	if (deviceClass == PHIDCLASS_VINT) {
		prc = Phidget_getHubPort(ph, &hubPort);
		CheckError(prc, "Getting HubPort", (PhidgetHandle *)&ph);

		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Hub Port: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, hubPort, channel);
	} else { //Not VINT
		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, channel);
	}

}

/**
* Displays info about the detached Phidget channel.
* Fired when a Phidget channel with onDetachHandler registered detaches
*
* @param ph The Phidget channel that fired the detach event
* @param *ctx Context pointer. Used to pass information to the event handler.
*/
static void CCONV onDetachHandler(PhidgetHandle ph, void *ctx) {
	PhidgetReturnCode prc; //Used to catch error codes from each Phidget function call
	Phidget_DeviceClass deviceClass;
	char* channelClassName;
	int32_t serialNumber;
	int32_t hubPort;
	int32_t channel;

	//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
	//www.phidgets.com/docs/Using_Multiple_Phidgets for information

	printf("\nDetach Event: ");
	/*
	* Get device information and display it.
	*/
	prc = Phidget_getDeviceSerialNumber(ph, &serialNumber);
	CheckError(prc, "Getting DeviceSerialNumber", (PhidgetHandle *)&ph);

	prc = Phidget_getChannel(ph, &channel);
	CheckError(prc, "Getting Channel", (PhidgetHandle *)&ph);

	prc = Phidget_getChannelClassName(ph, &channelClassName);
	CheckError(prc, "Getting Channel Class Name", (PhidgetHandle *)&ph);

	prc = Phidget_getDeviceClass(ph, &deviceClass);
	CheckError(prc, "Getting Device Class", (PhidgetHandle *)&ph);

	if (deviceClass == PHIDCLASS_VINT) {
		prc = Phidget_getHubPort(ph, &hubPort);
		CheckError(prc, "Getting HubPort", (PhidgetHandle *)&ph);

		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Hub Port: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, hubPort, channel);
	} else { //Not VINT
		printf("\n\t-> Channel Class: %s\n\t-> Serial Number: %d\n\t-> Channel %d\n\n", channelClassName, serialNumber, channel);
	}
}

/**
* Writes Phidget error info to stderr.
* Fired when a Phidget channel with onErrorHandler registered encounters an error in the library
*
* @param ph The Phidget channel that fired the error event
* @param *ctx Context pointer. Used to pass information to the event handler.
* @param errorCode the code associated with the error of enum type Phidget_ErrorEventCode
* @param *errorString string containing the description of the error fired
*/
static void CCONV onErrorHandler(PhidgetHandle ph, void *ctx, Phidget_ErrorEventCode errorCode, const char *errorString) {

	fprintf(stderr, "[Phidget Error Event] -> %s (%d)\n", errorString, errorCode);
}

/**
* Outputs the IR's most recently reported IR code.
* Fired when a IR channel receives a code
*
* @param ph The IR channel that fired the Code event
* @param *ctx Context pointer. Used to pass information to the event handler.
* @param code The reported code from the IR channel
* @param bitCount The number of bits in the IR code
* @param isRepeat Whether or not the code is a repeat of the previous code
*/
static void CCONV onCodeHandler(PhidgetIRHandle ph, void *ctx, const char* code, uint32_t bitCount, int isRepeat) {

	//If you are unsure how to use more than one Phidget channel with this event, we recommend going to
	//www.phidgets.com/docs/Using_Multiple_Phidgets for information

	printf("\n[Code Event] -> Code:     %s\n", code);
	printf("             -> bitCount: %d\n", bitCount);
	printf("             -> isRepeat: %d\n", isRepeat);
}


/**
* Outputs the IR's most recently learned IR code.
* Fired when a IR channel learns a code
*
* @param ph The IR channel that fired the Code event
* @param *ctx Context pointer. Used to pass information to the event handler.
* @param codeStr The learned code
* @param codeInfo Other information about the learned code
*/
static void CCONV
onLearnHandler(PhidgetIRHandle phid, void *userPtr, const char *codeStr, PhidgetIR_CodeInfoHandle codeInfo) {
	int i, toggleMaskBytes;
	char *encodingStr;
	char *lengthStr;

	printf("\n[Learn Event]");
	printf("\n\t-> Code Read: 0x%s", codeStr);
	printf("\n");

	encodingStr = "";
	switch (codeInfo->encoding) {
	case IR_ENCODING_UNKNOWN:
		encodingStr = "Unknown";
		break;
	case IR_ENCODING_SPACE:
		encodingStr = "Space";
		break;
	case IR_ENCODING_PULSE:
		encodingStr = "Pulse";
		break;
	case IR_ENCODING_BIPHASE:
		encodingStr = "BiPhase";
		break;
	case IR_ENCODING_RC5:
		encodingStr = "RC5";
		break;
	case IR_ENCODING_RC6:
		encodingStr = "RC6";
		break;
	default:
		encodingStr = "Unknown";
		break;
	}
	lengthStr = "";
	switch (codeInfo->length) {
	case IR_LENGTH_UNKNOWN:
		lengthStr = "Unknown";
		break;
	case IR_LENGTH_CONSTANT:
		lengthStr = "Constant";
		break;
	case IR_LENGTH_VARIABLE:
		lengthStr = "Variable";
		break;
	default:
		lengthStr = "Unknown";
		break;
	}

	printf("\t-> Learned Code Info:\n");
	printf("----------------------------------------------------\n");

	printf("Bit Count: %d\nEncoding: %s\nLength: %s\nGap: %d\nTrail: %d", codeInfo->bitCount, encodingStr, lengthStr, codeInfo->gap, codeInfo->trail);
	printf("Header: { %d, %d }\nOne: { %d, %d }\nZero: { %d, %d }\n", codeInfo->header[0], codeInfo->header[1], codeInfo->one[0], codeInfo->one[1], codeInfo->zero[0], codeInfo->zero[1]);
	printf("Repeat: {");
	for (i = 0; i < 26; i++) {
		if (codeInfo->repeat[i] == 0)
			break;
		if (i == 0)
			printf("%d", codeInfo->repeat[i]);
		else
			printf(", %d", codeInfo->repeat[i]);
	}
	printf("}\n");
	printf("MinRepeat: %d\n", codeInfo->minRepeat);
	printf("Toggle Mask: ");

	toggleMaskBytes = 0;
	if ((codeInfo->bitCount % 8) == 0)
		toggleMaskBytes = (codeInfo->bitCount / 8) + 0;
	else
		toggleMaskBytes = (codeInfo->bitCount / 8) + 1;

	for (i = 0; i < toggleMaskBytes; i++) {
		printf("0x%x, ", codeInfo->toggleMask[i]);
	}
	printf("\n");
	printf("Carrier Frequency: %d\nDuty Cycle: %lf\n", codeInfo->carrierFrequency, codeInfo->dutyCycle);
	printf("----------------------------------------------------\n");

}

/**
* Prints descriptions of the available events for this class
*/
void PrintEventDescriptions() {

	printf("\n--------------------\n"
		"\n  | Code events will call their associated function every time new IR code data is received from the device.\n"
		"  | Press ENTER once you have read this message.\n");

	printf(
		"\n  | Learn events will call their associated function every time new IR code data is learned.\n"
		"  | Press ENTER once you have read this message.\n");

	getchar();

	printf("\n--------------------\n");
}


/**
* Creates, configures, and opens a IR channel.
* Displays tag and tag lost events for 10 seconds
* Closes out IR channel
*
* @return 0 if the program exits successfully, 1 if it exits with errors.
*/
int main() {
	PhidgetIRHandle ch = NULL;
	ChannelInfo channelInfo; //Information from AskForDeviceParameters(). May be removed when hard-coding parameters.
	PhidgetReturnCode prc; //Used to catch error codes from each Phidget function call

	/*
	* Allocate a new Phidget Channel object
	*/

	prc = PhidgetIR_create(&ch);
	CheckError(prc, "Creating Channel", (PhidgetHandle *)&ch);

	/*
	* Set addressing parameters to specify which channel to open
	*/

	//You can safely remove this line and hard-code the parameters to make the program can start without user input
	AskForDeviceParameters(&channelInfo, (PhidgetHandle *)&ch);

	prc = Phidget_setDeviceSerialNumber((PhidgetHandle)ch, channelInfo.deviceSerialNumber);
	CheckError(prc, "Setting DeviceSerialNumber", (PhidgetHandle *)&ch);

	prc = Phidget_setHubPort((PhidgetHandle)ch, channelInfo.hubPort);
	CheckError(prc, "Setting HubPort", (PhidgetHandle *)&ch);

	prc = Phidget_setChannel((PhidgetHandle)ch, channelInfo.channel);
	CheckError(prc, "Setting Channel", (PhidgetHandle *)&ch);

	if (channelInfo.netInfo.isRemote) {
		prc = Phidget_setIsRemote((PhidgetHandle)ch, channelInfo.netInfo.isRemote);
		CheckError(prc, "Setting IsRemote", (PhidgetHandle *)&ch);
		if (channelInfo.netInfo.serverDiscovery) {
			prc = PhidgetNet_enableServerDiscovery(PHIDGETSERVER_DEVICEREMOTE);
			CheckEnableServerDiscoveryError(prc, (PhidgetHandle *)&ch);
		} else {
			prc = PhidgetNet_addServer("Server", channelInfo.netInfo.hostname,
				channelInfo.netInfo.port, channelInfo.netInfo.password, 0);
			CheckError(prc, "Adding Server", (PhidgetHandle *)&ch);
		}
	}

	/*
	* Add event handlers before calling open so that no events are missed.
	*/

	printf("\n--------------------------------------\n");
	printf("\nSetting OnAttachHandler...\n");
	prc = Phidget_setOnAttachHandler((PhidgetHandle)ch, onAttachHandler, NULL);
	CheckError(prc, "Setting OnAttachHandler", (PhidgetHandle *)&ch);

	printf("Setting OnDetachHandler...\n");
	prc = Phidget_setOnDetachHandler((PhidgetHandle)ch, onDetachHandler, NULL);
	CheckError(prc, "Setting OnDetachHandler", (PhidgetHandle *)&ch);

	printf("Setting OnErrorHandler...\n");
	prc = Phidget_setOnErrorHandler((PhidgetHandle)ch, onErrorHandler, NULL);
	CheckError(prc, "Setting OnErrorHandler", (PhidgetHandle *)&ch);

	//This call may be harmlessly removed
	PrintEventDescriptions();

	printf("Setting OnCodeHandler...\n");
	prc = PhidgetIR_setOnCodeHandler(ch, onCodeHandler, NULL);
	CheckError(prc, "Setting OnCodeHandler", (PhidgetHandle *)&ch);

	printf("Setting OnLearnHandler...\n");
	prc = PhidgetIR_setOnLearnHandler(ch, onLearnHandler, NULL);
	CheckError(prc, "Setting OnLearnHandler", (PhidgetHandle *)&ch);

	/*
	* Open the channel with a timeout
	*/
	printf("Opening and Waiting for Attachment...\n");
	prc = Phidget_openWaitForAttachment((PhidgetHandle)ch, 5000);
	CheckOpenError(prc, (PhidgetHandle *)&ch);

	/*
	* To find additional functionality not included in this example,
	* be sure to check the API for your device.
	*/

	printf("Sampling data for 10 seconds...\n");

	Sleep(10000);

	/*
	* Perform clean up and exit
	*/

	//clear the Code event handler
	prc = PhidgetIR_setOnCodeHandler(ch, NULL, NULL);
	CheckError(prc, "Clearing OnCodeHandler", (PhidgetHandle *)&ch);

	//clear the Learn event handler
	prc = PhidgetIR_setOnLearnHandler(ch, NULL, NULL);
	CheckError(prc, "Clearing OnLearnHandler", (PhidgetHandle *)&ch);

	printf("\nDone Sampling...\n");

	printf("Cleaning up...\n");
	prc = Phidget_close((PhidgetHandle)ch);
	CheckError(prc, "Closing Channel", (PhidgetHandle *)&ch);
	prc = PhidgetIR_delete(&ch);
	CheckError(prc, "Deleting Channel", (PhidgetHandle *)&ch);
	printf("\nExiting...\n");
	printf("Press ENTER to end program.\n");
	getchar();

	return 0;

}

