#include <stdio.h>
#include <stdlib.h>

#include <phidget22.h>

#ifndef _WIN32
#include <unistd.h>
#else
#include <Windows.h>
#endif

static void CCONV ssleep(int);

static void CCONV
onAttachHandler(PhidgetHandle phid, void *ctx) {
	PhidgetReturnCode res;
	int hubPort;
	int channel;
	int serial;

	res = Phidget_getDeviceSerialNumber(phid, &serial);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to get device serial number\n");
		return;
	}

	res = Phidget_getChannel(phid, &channel);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to get channel number\n");
		return;
	}

	res = Phidget_getHubPort(phid, &hubPort);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to get hub port\n");
		hubPort = -1;
	}

	if (hubPort == -1)
		printf("channel %d on device %d attached\n", channel, serial);
	else
		printf("channel %d on device %d hub port %d attached\n", channel, serial, hubPort);
}

static void CCONV
onDetachHandler(PhidgetHandle phid, void *ctx) {
	PhidgetReturnCode res;
	int hubPort;
	int channel;
	int serial;

	res = Phidget_getDeviceSerialNumber(phid, &serial);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to get device serial number\n");
		return;
	}

	res = Phidget_getChannel(phid, &channel);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to get channel number\n");
		return;
	}

	res = Phidget_getHubPort(phid, &hubPort);
	if (res != EPHIDGET_OK)
		hubPort = -1;

	if (hubPort != -1)
		printf("channel %d on device %d detached\n", channel, serial);
	else
		printf("channel %d on device %d hub port %d detached\n", channel, hubPort, serial);
}

static void CCONV
errorHandler(PhidgetHandle phid, void *ctx, Phidget_ErrorEventCode errorCode, const char *errorString) {

	fprintf(stderr, "Error: %s (%d)\n", errorString, errorCode);
}

static void CCONV
onCodeHandler(PhidgetIRHandle phid, void *userPtr, const char *codeStr, uint32_t bitCount, int repeat) {
	printf("\nIn Code Handler");
	printf("\nCode Read: 0x%s", codeStr);
	printf("\nBit Count: %d", bitCount);
	printf("\nRepeat: %d", repeat);

	printf("\n");
}

static void CCONV
onLearnHandler(PhidgetIRHandle phid, void *userPtr, const char *codeStr, PhidgetIR_CodeInfoHandle codeInfo) {
	int i, toggleMaskBytes;
	char *encodingStr;
	char *lengthStr;

	printf("\nIn Learn Handler");
	printf("\nCode Read: 0x%s", codeStr);
	printf("\n");

	encodingStr = "";
	switch (codeInfo->encoding)
	{
	case IR_ENCODING_UNKNOWN:
		encodingStr = "Unknown";
		break;
	case IR_ENCODING_SPACE:
		encodingStr = "Space";
		break;
	case IR_ENCODING_PULSE:
		encodingStr = "Pulse";
		break;
	case IR_ENCODING_BIPHASE:
		encodingStr = "BiPhase";
		break;
	case IR_ENCODING_RC5:
		encodingStr = "RC5";
		break;
	case IR_ENCODING_RC6:
		encodingStr = "RC6";
		break;
	default:
		encodingStr = "Unknown";
		break;
	}
	lengthStr = "";
	switch (codeInfo->length)
	{
	case IR_LENGTH_UNKNOWN:
		lengthStr = "Unknown";
		break;
	case IR_LENGTH_CONSTANT:
		lengthStr = "Constant";
		break;
	case IR_LENGTH_VARIABLE:
		lengthStr = "Variable";
		break;
	default:
		lengthStr = "Unknown";
		break;
	}

	printf("Learned Code Info:\n");
	printf("----------------------------------------------------\n");

	printf("Bit Count: %d\nEncoding: %s\nLength: %s\nGap: %d\nTrail: %d", codeInfo->bitCount, encodingStr, lengthStr, codeInfo->gap, codeInfo->trail);
	printf("Header: { %d, %d }\nOne: { %d, %d }\nZero: { %d, %d }\n", codeInfo->header[0], codeInfo->header[1], codeInfo->one[0], codeInfo->one[1], codeInfo->zero[0], codeInfo->zero[1]);
	printf("Repeat: {");
	for (i = 0; i < 26; i++)
	{
		if (codeInfo->repeat[i] == 0)
			break;
		if (i == 0)
			printf("%d", codeInfo->repeat[i]);
		else
			printf(", %d", codeInfo->repeat[i]);
	}
	printf("}\n");
	printf("MinRepeat: %d\n", codeInfo->minRepeat);
	printf("Toggle Mask: ");

	toggleMaskBytes = 0;
	if ((codeInfo->bitCount % 8) == 0)
		toggleMaskBytes = (codeInfo->bitCount / 8) + 0;
	else
		toggleMaskBytes = (codeInfo->bitCount / 8) + 1;

	for (i = 0; i < toggleMaskBytes; i++)
	{
		printf("%s", codeInfo->toggleMask);
	}
	printf("\n");
	printf("Carrier Frequency: %d\nDuty Cycle: %lf\n", codeInfo->carrierFrequency, codeInfo->dutyCycle);
	printf("----------------------------------------------------\n");

}


/*
* Creates and initializes the channel.
*/
static PhidgetReturnCode CCONV
initChannel(PhidgetHandle ch) {
	PhidgetReturnCode res;

	res = Phidget_setOnAttachHandler(ch, onAttachHandler, NULL);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to assign on attach handler\n");
		return (res);
	}

	res = Phidget_setOnDetachHandler(ch, onDetachHandler, NULL);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to assign on detach handler\n");
		return (res);
	}

	res = Phidget_setOnErrorHandler(ch, errorHandler, NULL);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to assign on error handler\n");
		return (res);
	}

	/*
	* Please review the Phidget22 channel matching documentation for details on the device
	* and class architecture of Phidget22, and how channels are matched to device features.
	*/

	/*
	* Specifies the serial number of the device to attach to.
	* For VINT devices, this is the hub serial number.
	*
	* The default is any device.
	*/
	// Phidget_setDeviceSerialNumber(ch, <YOUR DEVICE SERIAL NUMBER>); 

	/*
	* For VINT devices, this specifies the port the VINT device must be plugged into.
	*
	* The default is any port.
	*/
	// Phidget_setHubPort(ch, 0);

	/*
	* Specifies that the channel should only match a VINT hub port.
	* The only valid channel id is 0.
	*
	* The default is 0 (false), meaning VINT hub ports will never match
	*/
	// Phidget_setIsHubPortDevice(ch, 1);

	/*
	* Specifies which channel to attach to.  It is important that the channel of
	* the device is the same class as the channel that is being opened.
	*
	* The default is any channel.
	*/
	// Phidget_setChannel(ch, 0);

	/*
	* In order to attach to a network Phidget, the program must connect to a Phidget22 Network Server.
	* In a normal environment this can be done automatically by enabling server discovery, which
	* will cause the client to discovery and connect to available servers.
	*
	* To force the channel to only match a network Phidget, set remote to 1.
	*/
	// PhidgetNet_enableServerDiscovery(PHIDGETSERVER_DEVICE);
	// Phidget_setIsRemote(ch, 1);

	return (EPHIDGET_OK);
}

int
main(int argc, char **argv) {
	PhidgetIRHandle ch;
	PhidgetReturnCode res;
	const char *errs;

	/*
	* Enable logging to stdout
	*/
	PhidgetLog_enable(PHIDGET_LOG_INFO, NULL);

	res = PhidgetIR_create(&ch);
	if (res != EPHIDGET_OK) {
		fprintf(stderr, "failed to create IR channel\n");
		exit(1);
	}

	res = initChannel((PhidgetHandle)ch);
	if (res != EPHIDGET_OK) {
		Phidget_getErrorDescription(res, &errs);
		fprintf(stderr, "failed to initialize channel:%s\n", errs);
		exit(1);
	}

	res = PhidgetIR_setOnCodeHandler(ch, onCodeHandler, NULL);
	if (res != EPHIDGET_OK) {
		Phidget_getErrorDescription(res, &errs);
		fprintf(stderr, "failed to set code handler: %s\n", errs);
		goto done;
	}

	res = PhidgetIR_setOnLearnHandler(ch, onLearnHandler, NULL);
	if (res != EPHIDGET_OK) {
		Phidget_getErrorDescription(res, &errs);
		fprintf(stderr, "failed to set code handler: %s\n", errs);
		goto done;
	}

	/*
	* Open the channel synchronously: waiting a maximum of 5 seconds.
	*/
	res = Phidget_openWaitForAttachment((PhidgetHandle)ch, 5000);
	if (res != EPHIDGET_OK) {
		if (res == EPHIDGET_TIMEOUT) {
			printf("Channel did not attach after 5 seconds: please check that the device is attached\n");
		} else {
			Phidget_getErrorDescription(res, &errs);
			fprintf(stderr, "failed to open channel:%s\n", errs);
		}
		goto done;
	}

	printf("Gathering data for 10 seconds...\n");
	ssleep(10);

done:

	Phidget_close((PhidgetHandle)ch);
	PhidgetIR_delete(&ch);

	exit(res);
}

static void CCONV
ssleep(int tm) {
#ifdef _WIN32
	Sleep(tm * 1000);
#else
	sleep(tm);
#endif
}