#include <stdio.h>
#include <stdlib.h>

#include <phidget22.h>

#ifndef _WIN32
#include <unistd.h>
#else
#include <Windows.h>
#endif

static void CCONV ssleep(int);

void CCONV
onDictionaryAdd(PhidgetDictionaryHandle ch, void *ctx, const char *key, const char* value) {
	printf("------------\n");
	printf("Key Added: \nKey: %s\nValue: %s\n", key, value);
}


void CCONV
onDictionaryUpdate(PhidgetDictionaryHandle ch, void *ctx, const char *key, const char* value) {
	printf("------------\n");
	printf("Value Updated: \nKey: %s\nValue: %s\n", key, value);

}

void CCONV
onDictionaryRemove(PhidgetDictionaryHandle ch, void *ctx, const char *key) {
	printf("------------\n");
	printf("Key Removed: \nKey: %s\n", key);
}

void listKeys(PhidgetDictionaryHandle dict);

int
main(int argc, char **argv) {
	PhidgetDictionaryHandle dict;
	PhidgetReturnCode res;
	const char *errs;
	
	char result[20];

	printf("This example requires that you have already created a dictionary labeled\n");
	printf("\"testdict\" in the control panel for the server hosting your dicitonary.\n");

	PhidgetNet_enableServerDiscovery(PHIDGETSERVER_DEVICE);

	PhidgetDictionary_create(&dict);

	Phidget_setDeviceLabel(dict, "testdict");

	PhidgetDictionary_setOnAddHandler(dict, onDictionaryAdd, NULL);
	PhidgetDictionary_setOnUpdateHandler(dict, onDictionaryUpdate, NULL);
	PhidgetDictionary_setOnRemoveHandler(dict, onDictionaryRemove, NULL);

	printf("\nWaiting to discover dictionary...\n");

	res = Phidget_openWaitForAttachment(dict, 20000);
	if (res != EPHIDGET_OK) {
		if (res == EPHIDGET_TIMEOUT) {
			printf("Dicitonary did not attach after 20 seconds: please check that it is running\n");
			printf("on your local network.\n");
		} else {
			Phidget_getErrorDescription(res, &errs);
			fprintf(stderr, "failed to open channel:%s\n", errs);
		}
		goto done;
	}


	printf("Attached!\n");

	//Set can be used to set the a value for a key even if the key doesn't exist yet
	PhidgetDictionary_set(dict, "key1", "value");
	PhidgetDictionary_get(dict, "key1", result, 20);
	printf("Result 1: %s\n", result);

	ssleep(3);

	PhidgetDictionary_set(dict, "key1", "set 2");
	PhidgetDictionary_get(dict, "key1", result, 20);
	printf("Result 2: %s\n", result);

	ssleep(3);

	//Update can be used to update the value of pre-existing keys
	PhidgetDictionary_update(dict, "key1", "updated");
	PhidgetDictionary_get(dict, "key1", result, 20);
	printf("Result 3: %s\n", result);

	ssleep(3);

	//As you might expect, Remove removes keys from the dictionary
	printf("Removing key1\n");
	PhidgetDictionary_remove(dict, "key1");

	ssleep(3);

	//You can add keys explicitly, to only add them if they don't yet exist.
	PhidgetDictionary_add(dict, "key1", "added key1");
	PhidgetDictionary_get(dict, "key1", result, 20);
	printf("Result 4: %s\n", result);

	ssleep(3);

	printf("Removing key1\n");
	PhidgetDictionary_remove(dict, "key1");

	printf("\nListing Keys: \n");

	listKeys(dict);

	printf("\nActive demonstration complete.\n");
	printf("Now would be a good time to try out the events by running a second copy of this\n");
	printf("program, or using the control pannel example to edit key-value pairs in the\n");
	printf("test dictionary.\n\n");

	printf("\nPress Enter to end anytime...\n");
	getchar();

done:

	Phidget_close((PhidgetHandle)dict);
	PhidgetDictionary_delete(&dict);

	exit(res);
}

void listKeys(PhidgetDictionaryHandle dict) {
	int len;
	char* start = NULL;
	char keysbuffer[200] = { 0 };
	char* keyslist[20] = { 0 };
	int keysindex = 0;

	do {
		PhidgetDictionary_scan(dict, start, keysbuffer, 200);
		len = strlen(keysbuffer);
		keyslist[keysindex] = strtok(keysbuffer, "\n");
		while (keyslist[keysindex] != NULL) {
			printf("%s\n", keyslist[keysindex]);
			keysindex++;
			keyslist[keysindex] = strtok(NULL, "\n");
		}

		start = keyslist[keysindex - 1];
	} while (len > 0);
}

static void CCONV
ssleep(int tm) {
#ifdef _WIN32
	Sleep(tm * 1000);
#else
	sleep(tm);
#endif
}
