﻿/*
This example starts by creating an DataAdapter object.

Before opening the DataAdapter object we register event and error handlers. ( Check the User Guide of your product
for list of events and errors ).

Then we open the the DataAdapter object with a specified serial number, on a specified channel (and a specified hub port, if necessary)

Once the "attached" event is fired, the OnAttachedEventHandler sends out the "must set" parameters (none for most devices in this class)

Once initialized, this program will send a packet to the master indicating its readiness.

The event handlers get called each time their respective event is fired. This program will use them to simply display the pertinent information, 
and reply with every byte incremented from one higher than the last received byte. 

Last but not least, the device is properly closed.

Note: When using Phidgets commands that communicate with the device, it is prudent to check the return value.
A return value of 0 means the function was successful but there are a number of descriptive error codes that might also be returned.
For a full list of error codes and what they mean check http://www.phidgets.com/docs/General_Phidget_Programming#Software_Errors
*/

#include <stdio.h>
#include <phidget22.h>

int CheckReturn(char* functionDesc, const int returnValue);

//the attach handler will run when the library detects that the device has been connected to your computer.
void CCONV OnDataAdapterAttachedEventHandler(PhidgetHandle Device, void *userPtr)
{
	int channel, port, serial;
	Phidget_DeviceClass devClass;

	PhidgetDataAdapterHandle DataAdapter = (PhidgetDataAdapterHandle)Device;

	printf("\n\nAttach Detected");

	//Determine if the device uses a hub
	CheckReturn("Getting Device Class",
		Phidget_getDeviceClass(Device, &devClass));

	CheckReturn("Getting Attached Channel",
		Phidget_getChannel(Device, &channel));

	CheckReturn("Getting Attached Serial Number",
		Phidget_getDeviceSerialNumber(Device, &serial));

	if (devClass == PHIDCLASS_VINT)
	{
		CheckReturn("Getting Attached Hub Port",
			Phidget_getHubPort(Device, &port));
		printf("\n\nDataAdapter Attach Event on Port %d Channel %d", port, channel);
		printf("\nHub Serial Number: %d", serial);
	}
	else
	{
		printf("\n\nDataAdapter Attach Event with USB Device, Channel %d", channel);
		printf("\nDevice Serial Number: %d", serial);
	}

	printf("\n");

}

void inputDataHandler(PhidgetDataAdapterHandle dataAdapter, void *ctx, const uint8_t data[48], uint8_t arrayCount, uint8_t overrun)
{
	uint8_t outData[48];
	
	printf("\n--------------------------------------");
	if (overrun)
		printf("\nOverrun has occured!");
	for (int i = 0; i < arrayCount; i++)
		printf("\n%d", data[i]);

	for (int i = 0; i < arrayCount; i++)
		outData[i] = data[i] + arrayCount;

	CheckReturn("Sending Reply",
		PhidgetDataAdapter_sendData(dataAdapter, outData, arrayCount));

}


//the detach event will run when the library detects that the device has become disconnected or has otherwise lost communication with the computer
void CCONV OnDetachedEventHandler(PhidgetHandle device, void *userPtr)
{
	int channel, port;
	Phidget_DeviceClass devClass;
	printf("\nDetach Detected");

	CheckReturn("Getting Channel",
		Phidget_getChannel(device, &channel));
	CheckReturn("Getting Device Class",
		Phidget_getDeviceClass(device, &devClass));

	if (devClass == PHIDCLASS_VINT)
	{
		CheckReturn("Getting Hub Port",
			Phidget_getHubPort(device, &port));

		printf("\nDetach Event on Port %d Channel %d", port, channel);
	}
	else
	{
		printf("\nDetach Event on Channel %d", channel);
	}
}

//the error handler will run when something problematic happens
void CCONV ErrorHandler(PhidgetHandle phid, void *userPtr, Phidget_ErrorEventCode errorCode, const char *errorString)
{
	printf("\nError code : %d \n\t Error String : %s ", errorCode, errorString);
}


int CheckReturn(char* functionDesc, const int returnValue)
{
	const char* errorString;

	printf("\n%s", functionDesc);
	if (returnValue != EPHIDGET_OK)
	{
		Phidget_getErrorDescription(returnValue, &errorString);
		printf("\n%s", errorString);
	}

	return returnValue;
}

int main()
{
	int serial, channel, port;
	PhidgetDataAdapterHandle DataAdapter;

	//uncomment the following function if you want detailed log information from core
	//PhidgetLog_enable(PHIDGET_LOG_INFO, NULL);

	//create the object that the physcial device will be associated with
	CheckReturn("Creating DataAdapter Device",
		PhidgetDataAdapter_create(&DataAdapter));

	//register various useful handlers
	CheckReturn("Setting up Attach Event Hadler",
		Phidget_setOnAttachHandler((PhidgetHandle)DataAdapter, OnDataAdapterAttachedEventHandler, NULL));
	CheckReturn("Setting up Detach Event Handler",
		Phidget_setOnDetachHandler((PhidgetHandle)DataAdapter, OnDetachedEventHandler, NULL));
	CheckReturn("Setting up Error Handler",
		Phidget_setOnErrorHandler((PhidgetHandle)DataAdapter, ErrorHandler, NULL));

	PhidgetDataAdapter_setOnDataReceivedHandler(DataAdapter, inputDataHandler, NULL);

	//Uncomment this define statement if the device is to be run remotely over a network
	//#define REMOTE
#ifdef REMOTE
	printf("\nOpening DataAdapter over network.");

	//Enable MDNS to find Phidgets over a local network
	CheckReturn("Enabling MDNS",
		PhidgetNet_enableMDNS(PHIDGETSERVER_DEVICE));

	Phidget_setIsRemote((PhidgetHandle)DataAdapter, 1);
#else
	printf("\nOpening DataAdapter locally.");
#endif

	//You can set the device or hub serial number to match your sscenario as required. -1 is a wildcard and will work for any device.
	CheckReturn("Setting the Serial Number",
		Phidget_setDeviceSerialNumber((PhidgetHandle)DataAdapter, -1));

	//The following command will work just as well as setDeviceSerialNumber, but may be clearer where a hub is involved
	//Phidget_setHubSerialNumber((PhidgetHandle)DataAdapter, -1);

	//If a hub is being used, you must set which port the device is on. Otherwise, this step may be ignored.
	CheckReturn("Setting the Hub Port",
		Phidget_setHubPort((PhidgetHandle)DataAdapter, 0));

	//Set this value to the channel being used on the device.
	CheckReturn("Setting the Channel",
		Phidget_setChannel((PhidgetHandle)DataAdapter, 0));

	//print the parameters onscreen to see them in case something goes wrong
	CheckReturn("Getting requested Serial Number",
		Phidget_getDeviceSerialNumber((PhidgetHandle)DataAdapter, &serial));
	CheckReturn("Getting requested Hub Port",
		Phidget_getHubPort((PhidgetHandle)DataAdapter, &port));
	CheckReturn("Getting requested Channel",
		Phidget_getChannel((PhidgetHandle)DataAdapter, &channel));

	printf("\n\nThis program is looking for an DataAdapter with:");
	printf("\n\nSerial: %d   Port: %d   Channel: %d", serial, port, channel);

	//Use the wait for attachment variant of Phidget_open if you want to prevent your program from executing before device is attached.
	//The return value of this function can be used to check if device successfully attached before the timeout expired.
	printf("\n\nOpening and waiting for attachment for 5 seconds");
	CheckReturn("",
		Phidget_openWaitForAttachment((PhidgetHandle)DataAdapter, 5000));


#define size 4
	uint8_t data[size];

	for (int i = 0; i < size; i++)
		data[i] = i + 1;

	CheckReturn("Sending data",
		PhidgetDataAdapter_sendData(DataAdapter, data, size));

	//From here the program is event driven, and will respond to data from the SPI master device
	_getch();

	//When the program terminates we need to make sure to dispose of the device correctly.  If we don't, strange things can happen and memory leaks can occur.
	CheckReturn("Closing DataAdapter",
		Phidget_close((PhidgetHandle)DataAdapter));

	CheckReturn("Deleting DataAdapter",
		PhidgetDataAdapter_delete(&DataAdapter));

	return 0;
}
