//======================================================================
// pddPhidget.cpp - Low level USB interface code
//
// Author: Patrick McNeil
// Based On: pddVClas.h by Douglas Boling
//======================================================================
#include "stdafx.h"
#include <windows.h>                // For all that Windows stuff
#include <usbdi.h>					// USB includes
#include <usb100.h>					// USB includes
#include <usbclient.h>				// USB client driver helper code

#include "PhidgetConstants.h"

#include "phidgetsdk.h"				// IOCTL defs for driver
#include "phidget.h" 				// Local driver includes
#include "pddphidget.h"				// PDD defs


//---------------------------------------------------------------------------------------
//
//
int pdd_DrvInit (PDRVCONTEXT pDrv) 
{
	LPCUSB_DEVICE devInfo;
    USB_TRANSFER hTransfer;
	BYTE pbBuffer[255];
	DWORD const USB_REPORT_DESCRIPTOR_TYPE = 0x22;
    DWORD bTransferred;
	int i, first=1;
    USB_ERROR usbErr;
	DWORD dw, dwErr;

	memset(pbBuffer, 0, 255);

	//DebugBreak();

	devInfo = pDrv->lpUsbFuncs->lpGetDeviceInfo(pDrv->hDevice);

	pDrv->attr.ProductID = devInfo->Descriptor.idProduct;
	pDrv->attr.VendorID = devInfo->Descriptor.idVendor;
	pDrv->attr.VersionNumber = devInfo->Descriptor.bcdDevice;

	//Open a pipe for Interrupt reads
	//TODO: verify that we get a valid pipe
	if(!pDrv->hPipe)
	{
		pDrv->hPipe = pDrv->lpUsbFuncs->lpOpenPipe(pDrv->hDevice, 
			&devInfo->lpActiveConfig->lpInterfaces[pDrv->attr.InterfaceNumber].lpEndpoints->Descriptor);
	}

	USB_DEVICE_REQUEST DeviceRequest;

    DeviceRequest.bmRequestType = USB_REQUEST_DEVICE_TO_HOST |
            USB_REQUEST_STANDARD | USB_REQUEST_FOR_INTERFACE;
    DeviceRequest.bRequest = USB_REQUEST_GET_DESCRIPTOR;
    DeviceRequest.wValue = ((WORD)USB_REPORT_DESCRIPTOR_TYPE) << 8;
    DeviceRequest.wIndex = pDrv->attr.InterfaceNumber;
    DeviceRequest.wLength = sizeof(pbBuffer);

    hTransfer = pDrv->lpUsbFuncs->lpIssueVendorTransfer(pDrv->hDevice, NULL, NULL,
		USB_IN_TRANSFER, &DeviceRequest, pbBuffer, NULL);

    if (hTransfer != NULL) {
        GetTransferStatus(pDrv->lpUsbFuncs, hTransfer, 
            &bTransferred, &usbErr);
        CloseTransferHandle(pDrv->lpUsbFuncs, hTransfer);

		switch(usbErr)
		{
			case USB_NO_ERROR:
			case USB_DATA_UNDERRUN_ERROR:
				break;
			case USB_STALL_ERROR:
				dwErr = ERROR_NOT_SUPPORTED;
				goto EXIT;
			case USB_DEVICE_NOT_RESPONDING_ERROR:
				dwErr = ERROR_DEVICE_NOT_CONNECTED;
				goto EXIT;
			default:
				dwErr = ERROR_GEN_FAILURE;
				goto EXIT;
		}
    }
    else {
        dwErr = GetLastError();
		if(dwErr == ERROR_SUCCESS)
			dwErr = ERROR_GEN_FAILURE;
        goto EXIT;
    }

    dwErr = ERROR_SUCCESS;

	for(i=10;i<bTransferred;i++) {
		if(pbBuffer[i]==0x81 && pbBuffer[i-2]==0x95)
			pDrv->attr.InputReportByteLength=pbBuffer[i-1];
		if(pbBuffer[i]==0x91 && pbBuffer[i-2]==0x95)
			pDrv->attr.OutputReportByteLength=pbBuffer[i-1];
	}

EXIT:
    if (dwErr != ERROR_SUCCESS) {
        DEBUGMSG(ZONE_WARNING, (_T("Error getting hid descriptor. Err=%u\r\n"),
            dwErr));
    }
    
    return dwErr; 
}

//---------------------------------------------------------------------------------------
//
//
int pdd_DrvUninit (PDRVCONTEXT pDrv) 
{
	BOOL result;
	//close the pipe
	if(pDrv->hPipe)
	{
		result = pDrv->lpUsbFuncs->lpClosePipe(pDrv->hPipe);
		pDrv->hPipe = NULL;
	}
	return result;
}

//---------------------------------------------------------------------------------------
// pdd_DrvOpen - Called when driver is opened
//
int pdd_DrvOpen (PDRVCONTEXT pDrv) 
{
	return 1;
}

//---------------------------------------------------------------------------------------
// pdd_DrvClose - Called when driver is closed
//
int pdd_DrvClose (PDRVCONTEXT pDrv) 
{
	BOOL result;
	//abort any transfers
	if(pDrv->hPipe)
	{
		result = pDrv->lpUsbFuncs->lpAbortPipeTransfers(pDrv->hPipe, 0);
	}
	return result;
}


//---------------------------------------------------------------------------------------
// pdd_SetTagString - Sets the tag
//
int pdd_SetTagString (PDRVCONTEXT pDrv, PPHIDDEV_STRING pdwVal, PDWORD pBytes) 
{
	int rc;
    USB_TRANSFER hTransfer;
    DWORD dwErr;
    DWORD bTransferred;
    USB_ERROR usbErr;

	DEBUGMSG (ZONE_FUNC | ZONE_FEATURE, 
	          (DTAG TEXT("pdd_SetParameter++  Val: %s\r\n"), pdwVal->string));

    hTransfer = pDrv->lpUsbFuncs->lpSetDescriptor(pDrv->hDevice, NULL, NULL,
		0, USB_STRING_DESCRIPTOR_TYPE, 4, 
		0x0409, pdwVal->size, pdwVal->string); //TODO: the langid may be wrong

    if (hTransfer != NULL) {
        GetTransferStatus(pDrv->lpUsbFuncs, hTransfer, 
            &bTransferred, &usbErr);
        CloseTransferHandle(pDrv->lpUsbFuncs, hTransfer);

        if (usbErr != USB_NO_ERROR) {
            if (usbErr == USB_STALL_ERROR) {
                dwErr = ERROR_NOT_SUPPORTED;
            }        
            else {
                dwErr = ERROR_GEN_FAILURE;
            }

            goto EXIT;
        }

		if(bTransferred != pdwVal->size)
		{
			dwErr = ERROR_NOT_SUPPORTED;
			goto EXIT;
		}
    }
    else {
        dwErr = GetLastError();
        goto EXIT;
    }

    dwErr = ERROR_SUCCESS;
	*pBytes = bTransferred;

	DEBUGMSG (ZONE_FUNC | ZONE_FEATURE, 
	          (DTAG TEXT("pdd_SetParameter--  rc %d\r\n"), rc));


EXIT:
    if (dwErr != ERROR_SUCCESS) {
        DEBUGMSG(ZONE_WARNING, (_T("Error getting hid descriptor. Err=%u\r\n"),
            dwErr));
    }

    return dwErr;  
}

//---------------------------------------------------------------------------------------
// pdd_Read - reads from the device (interrupt)
// returns bytes read
//
DWORD pdd_Read (PDRVCONTEXT pDrv, LPVOID pBuffer, DWORD dwCount, PDWORD pbTransferred)
{
    USB_TRANSFER hTransfer;
    USB_ERROR usbErr;
	DWORD dwErr;
	DEBUGMSG (ZONE_FUNC | ZONE_FEATURE, 
	          (DTAG TEXT("pdd_Read++\r\n")));

    hTransfer = pDrv->lpUsbFuncs->lpIssueInterruptTransfer(pDrv->hPipe, NULL, NULL,
		USB_IN_TRANSFER, dwCount, pBuffer, NULL);

    if (hTransfer != NULL) {
        GetTransferStatus(pDrv->lpUsbFuncs, hTransfer, 
            pbTransferred, &usbErr);
        CloseTransferHandle(pDrv->lpUsbFuncs, hTransfer);

		switch(usbErr)
		{
			case USB_NO_ERROR:
				break;
			case USB_STALL_ERROR:
				dwErr = ERROR_NOT_SUPPORTED;
				goto EXIT;
			case USB_DEVICE_NOT_RESPONDING_ERROR:
				dwErr = ERROR_DEVICE_NOT_CONNECTED;
				goto EXIT;
			default:
				dwErr = ERROR_GEN_FAILURE;
				goto EXIT;
		}
    }
    else {
        dwErr = GetLastError();
		if(dwErr == ERROR_SUCCESS)
			dwErr = ERROR_GEN_FAILURE;
        goto EXIT;
    }

    dwErr = ERROR_SUCCESS;

EXIT:
    if (dwErr != ERROR_SUCCESS) {
        DEBUGMSG(ZONE_WARNING, (_T("Error Reading Err=%u\n"),
            dwErr));
    }
    
    return dwErr;
}

DWORD pdd_Write (PDRVCONTEXT pDrv, LPVOID Buffer, DWORD dwCount, PDWORD pbTransferred)
{                     
	int rc;
	USB_DEVICE_REQUEST req;
    USB_TRANSFER hTransfer;
    USB_ERROR usbErr;
	DWORD dwErr;
	DEBUGMSG (ZONE_FUNC | ZONE_FEATURE,
	          (DTAG TEXT("pdd_Write++\r\n")));

	req.bmRequestType = USB_REQUEST_HOST_TO_DEVICE | USB_REQUEST_CLASS | USB_REQUEST_FOR_INTERFACE;
	req.bRequest = USB_REQUEST_SET_CONFIGURATION;
	req.wValue   = 0x0200;
	req.wIndex   = pDrv->attr.InterfaceNumber;
	req.wLength  = dwCount;

    hTransfer = pDrv->lpUsbFuncs->lpIssueVendorTransfer(pDrv->hDevice, NULL, NULL,
		USB_OUT_TRANSFER, &req, Buffer, NULL);

    if (hTransfer != NULL) {
        GetTransferStatus(pDrv->lpUsbFuncs, hTransfer, 
            pbTransferred, &usbErr);
        CloseTransferHandle(pDrv->lpUsbFuncs, hTransfer);

		switch(usbErr)
		{
			case USB_NO_ERROR:
				break;
			case USB_STALL_ERROR:
				dwErr = ERROR_NOT_SUPPORTED;
				goto EXIT;
			case USB_DEVICE_NOT_RESPONDING_ERROR:
				dwErr = ERROR_DEVICE_NOT_CONNECTED;
				goto EXIT;
			default:
				dwErr = ERROR_GEN_FAILURE;
				goto EXIT;
		}
    }
    else {
        dwErr = GetLastError();
		if(dwErr == ERROR_SUCCESS)
			dwErr = ERROR_GEN_FAILURE;
        goto EXIT;
    }

    dwErr = ERROR_SUCCESS;

EXIT:
    if (dwErr != ERROR_SUCCESS) {
        DEBUGMSG(ZONE_WARNING, (_T("Error Writing Err=%u\n"),
            dwErr));
    }
    
    return dwErr;
}

// Get a device string. For predefined types see hiddi.h.
// Call with pszBuffer == NULL to get the character count required 
// (then add 1 for the NULL terminator).
// This function is from the WindowsCE Hid Driver
DWORD
Pdd_GetString(
    PDRVCONTEXT		pDrv,
    DWORD           dwIdx,     // Only used with stringType == HID_STRING_INDEXED
    LPWSTR          pszBuffer, // Set to NULL to get character count
    DWORD           cchBuffer, // Count of chars that will fit into pszBuffer
                               // including the NULL terminator.
    PDWORD          pcchActual // Count of chars in the string NOT including
                               // the NULL terminator
    )
{

    const DWORD CB_STRING_DESCRIPTOR_MAX = 0xff;

    union {
        BYTE rgbBuffer[CB_STRING_DESCRIPTOR_MAX];
        USB_STRING_DESCRIPTOR StringDesc;
    } StringDescriptor;
    
    PCUSB_DEVICE pDeviceInfo = NULL;
    DWORD dwErr;
    WORD wLangId;
//    DWORD cchToCopy;
    DWORD dwBytesTransferred;
    BYTE bIdx;

    // Mdd guarantees the following
    DEBUGCHK(pcchActual != NULL);

	//indexes strings with a byte. Make sure dwIdx can be represented
    // in a byte.
    if (dwIdx > UCHAR_MAX) {
        DEBUGMSG(ZONE_ERROR, (_T("String index 0x%x is too large\r\n"),
            dwIdx));
        dwErr = ERROR_INVALID_PARAMETER;
        goto EXIT;
    }

    bIdx = (BYTE) dwIdx;

    // Get the Zero string descriptor to determine which LANGID to use.
    // We just use the first LANGID listed.
    dwErr = GetStringDescriptor(pDrv, 0, 0, (PBYTE) &StringDescriptor,
        4, &dwBytesTransferred);
    if (dwErr != ERROR_SUCCESS) {
        goto EXIT;
    }

    DEBUGCHK(StringDescriptor.StringDesc.bLength >= sizeof(USB_STRING_DESCRIPTOR));
    DEBUGCHK(StringDescriptor.StringDesc.bDescriptorType == USB_STRING_DESCRIPTOR_TYPE);
    wLangId = StringDescriptor.StringDesc.bString[0];

    // Get the string descriptor for the first LANGID
    dwErr = GetStringDescriptor(pDrv, bIdx, wLangId, (PBYTE) &StringDescriptor,
        sizeof(StringDescriptor), &dwBytesTransferred);
    if (dwErr != ERROR_SUCCESS) {
        goto EXIT;
    }

    __try {
        // Copy the character count and string into the user's buffer
        *pcchActual = (StringDescriptor.StringDesc.bLength - 2) / sizeof(WCHAR); // Does not include NULL
        if (pszBuffer != NULL) {
            StringCchCopyN(pszBuffer, cchBuffer, 
                StringDescriptor.StringDesc.bString, *pcchActual);
        }
    }
    __except (EXCEPTION_EXECUTE_HANDLER) {
        DEBUGMSG(ZONE_ERROR, (_T("Exception writing to user buffer\r\n")));
        dwErr = ERROR_INVALID_PARAMETER;
    }

EXIT:
    return dwErr;
}
//-----------------------------------------------------------------------
// GetStringDescriptor - Returns an indexed string
// 
// Retrieves the requested string descriptor and validates it.
// This function is from the WindowsCE Hid Driver
DWORD
GetStringDescriptor(
    PDRVCONTEXT pDrv,
    BYTE bIdx,
    WORD wLangId,
    PBYTE pbBuffer,
    WORD  cbBuffer,
    PDWORD pbTransferred
    )
{
    PUSB_STRING_DESCRIPTOR pStringDesc = (PUSB_STRING_DESCRIPTOR) pbBuffer;
    USB_ERROR usbErr;
    USB_TRANSFER hTransfer;
    DWORD dwErr;
    
    hTransfer = pDrv->lpUsbFuncs->lpGetDescriptor(pDrv->hDevice, NULL, NULL,
        NULL, USB_STRING_DESCRIPTOR_TYPE, bIdx, 
        wLangId, cbBuffer, pbBuffer);

    if (hTransfer != NULL) {
        GetTransferStatus(pDrv->lpUsbFuncs, hTransfer, 
            pbTransferred, &usbErr);
        CloseTransferHandle(pDrv->lpUsbFuncs, hTransfer);

		switch(usbErr)
		{
			case USB_NO_ERROR:
			case USB_DATA_UNDERRUN_ERROR:
				break;
			case USB_STALL_ERROR:
				dwErr = ERROR_NOT_SUPPORTED;
				goto EXIT;
			default:
				dwErr = ERROR_GEN_FAILURE;
				goto EXIT;
		}
    }
    else {
        dwErr = GetLastError();
        goto EXIT;
    }
    
    // We've got a descriptor. Is it valid?
    if ( (*pbTransferred < (sizeof(USB_STRING_DESCRIPTOR) - sizeof(pStringDesc->bString))) || 
         (pStringDesc->bDescriptorType != USB_STRING_DESCRIPTOR_TYPE) ) { 
        DEBUGCHK(FALSE); // The device returned something strange.
        dwErr = ERROR_GEN_FAILURE;
        goto EXIT;
    }

    dwErr = ERROR_SUCCESS;

EXIT:
    if (dwErr != ERROR_SUCCESS) {
        DEBUGMSG(ZONE_WARNING, (_T("Error getting string descriptor %u, %u. Err=%u\r\n"),
            bIdx, wLangId, dwErr));
    }
    
    return dwErr;    
}