//======================================================================
// USBCode - USB support entry points for driver
//
// Author: Patrick McNeil
// Based On: USBCode.h by Douglas Boling
//======================================================================
#include "stdafx.h"
#include <windows.h>                // For all that Windows stuff
#include <USBdi.h>					// USB includes
#include <usb100.h>					// USB includes
#include <usbclient.h>				// USB client driver helper code

#include "PhidgetConstants.h"

#include "phidgetsdk.h"				// IOCTL defs for driver
#include "phidget.h" 				// Local driver includes


// function to dump USB info
BOOL USBDeviceNotificationCallback (LPVOID lpvNotifyParameter, DWORD dwCode,
									LPDWORD* dwInfo1, LPDWORD* dwInfo2,
									LPDWORD* dwInfo3, LPDWORD* dwInfo4);

//======================================================================
// USBDeviceAttach - Called when Host controller wants to load the driver
// 
BOOL USBDeviceAttach (USB_HANDLE hDevice, LPCUSB_FUNCS lpUsbFuncs,
                      LPCUSB_INTERFACE lpInterface, LPCWSTR szUniqueDriverId,
                      LPBOOL fAcceptControl,
                      LPCUSB_DRIVER_SETTINGS lpDriverSettings, DWORD dwUnused)
{
    WCHAR wsSubClassRegKey[] = CLIENT_REGKEY_SZ;
	DWORD dwContext;

    DEBUGMSG (ZONE_USBLOAD, (DTAG TEXT("USBDeviceAttach++\r\n")));

	// Default is that we won't accept this device
	*fAcceptControl = FALSE;

	// Create and initialize the driver instance structure.  Alloc
	// now since the PDD will need some of the info.  Free if PDD
	// rejects the interface.
	PDRVCONTEXT pDrv = (PDRVCONTEXT)LocalAlloc (LPTR, sizeof (DRVCONTEXT));
	if (pDrv == 0)
	{ 
		SetLastError (ERROR_NOT_ENOUGH_MEMORY);
		return FALSE;
	}
    pDrv->dwSize = sizeof (DRVCONTEXT);
	pDrv->hDevice = hDevice;
	pDrv->lpUsbFuncs = lpUsbFuncs;
	InitializeCriticalSection (&pDrv->csDCall);
	InitializeCriticalSection (&pDrv->csRead);
	InitializeCriticalSection (&pDrv->csWrite);
	//pDrv->hUnload = CreateEvent (NULL, FALSE, FALSE, NULL);
	//pDrv->fUnloadFlag = FALSE;
	pDrv->dwPddContext = dwContext;
	pDrv->hPipe = 0;
	//Save the interface number
	pDrv->attr.InterfaceNumber = lpInterface->Descriptor.bInterfaceNumber;

	// Load us as a strean device driver...
	pDrv->hStreamDevice = ActivateDevice (wsSubClassRegKey, (DWORD)pDrv);

	if (pDrv->hStreamDevice) 
	{
		// register for USB callbacks
		if (lpUsbFuncs->lpRegisterNotificationRoutine (hDevice, USBDeviceNotificationCallback, pDrv))
		{
			// Accept this device as our own
			*fAcceptControl = TRUE;

		} else
			DEBUGMSG (ZONE_ERROR, (DTAG TEXT("Can't register for USB Callbacks! rc=%d\r\n"),
			          GetLastError()));
	} else
		DEBUGMSG (ZONE_ERROR, (DTAG TEXT("Can't activate stream device! rc=%d\r\n"),
		          GetLastError()));

    DEBUGMSG (ZONE_USBLOAD, (DTAG TEXT("USBDeviceAttach--\r\n")));
	return TRUE;
}
//======================================================================
// USBDeviceNotificationCallback - This routine is called by the USB
// host controller when a USB event occurs that the driver needs to
// know about.
// TODO: is there a way to call into userspace so they can receive this notification as well?
// 
BOOL USBDeviceNotificationCallback (LPVOID lpvNotifyParameter, DWORD dwCode,
									LPDWORD* dwInfo1, LPDWORD* dwInfo2,
									LPDWORD* dwInfo3, LPDWORD* dwInfo4)
{
	if (dwCode == USB_CLOSE_DEVICE)
	{
		// Get the pointer to our device context.
		PDRVCONTEXT pDrv = (PDRVCONTEXT) lpvNotifyParameter;

		// Set flags and handles to notify stream driver its about to unload.
		//TODO: what exactly is going on here...???
		//pDrv->fUnloadFlag = TRUE;
		//SetEvent (pDrv->hUnload);
		//Sleep (0);

		// Unload the stream driver
		DeactivateDevice (pDrv->hStreamDevice);

		Sleep (1000);
		LocalFree (pDrv);
		return TRUE;
	}
	return FALSE;
}
//======================================================================
// USBInstallDriver - Called by Host controller to have the driver
// register itself.  This call is made in response to the user
// entering the driver's DLL name in the "Unknown Device" message box.
// 

BOOL USBInstallDriver (LPCWSTR szDriverLibFile)
{
    WCHAR wsUsbDeviceID[] = CLASS_NAME_SZ;
    WCHAR wsSubClassRegKey[] = CLIENT_REGKEY_SZ;
    USB_DRIVER_SETTINGS usbDriverSettings = { DRIVER_SETTINGS };
	BOOL bSuccess;
	LPREGISTER_CLIENT_DRIVER_ID lpfnRegisterClientDriverID;
	LPREGISTER_CLIENT_SETTINGS lpfnRegisterClientSettings;

    DEBUGMSG (ZONE_USBLOAD, (DTAG TEXT("USBInstallDriver++\r\n")));

	HINSTANCE hUSBD = LoadLibrary (TEXT("usbd.dll"));
	if (hUSBD == 0)
	{
	    DEBUGMSG (ZONE_ERROR, (DTAG TEXT("Can't load USBD.DLL\r\n")));
		return FALSE;
	}

	// Explicitly link to USBD DLL.  
	lpfnRegisterClientDriverID = (LPREGISTER_CLIENT_DRIVER_ID) GetProcAddress (hUSBD, TEXT("RegisterClientDriverID"));
	lpfnRegisterClientSettings = (LPREGISTER_CLIENT_SETTINGS) GetProcAddress (hUSBD, TEXT("RegisterClientSettings"));
	if ((lpfnRegisterClientDriverID == 0) || (lpfnRegisterClientSettings == 0))
	{
		FreeLibrary (hUSBD);
		return FALSE;
	}
	//
	// register with USBD
	//	 
	bSuccess = (lpfnRegisterClientDriverID) (wsUsbDeviceID);
	if (!bSuccess) 
	{
		DEBUGMSG (ZONE_ERROR, (TEXT("RegisterClientDriverID error:%d\n"), GetLastError()));
		FreeLibrary (hUSBD);
		return FALSE;
	}
	
	// Call USBD to create registry entries for USB client registration for each Phidget
	int i;
	for(i=PHID_PID_MIN;i<=PHID_PID_MAX;i++)
	{
		usbDriverSettings.dwVendorId = PHID_VID;
		usbDriverSettings.dwProductId = i;

		bSuccess = (lpfnRegisterClientSettings) (szDriverLibFile, wsUsbDeviceID, 
												 NULL, &usbDriverSettings);
		if (!bSuccess) 
		{
			DEBUGMSG (ZONE_ERROR, (TEXT("RegisterClientSettings error:%d\n"), GetLastError()));
			FreeLibrary (hUSBD);
			return FALSE;
		}
	}

	//
	// The USB host controller driver has a utility function to make it
	// simple to create (and later query) the proper registery entries 
	// for loading a stream driver.  First, we create a table of registry 
	//values we want to set.
	//
    REG_VALUE_DESCR usbPhidKeyValues[] = {
        (TEXT("Dll")),               REG_SZ,    0, (PBYTE)(DRIVER_NAME),
        (TEXT("Prefix")),            REG_SZ,    0, (PBYTE)(DEVICE_PREFIX),
        NULL, 0, 0, NULL
    };

    // Set the default in the registry values.  This routine is in the 
	// USBClient library that we link to.
    bSuccess = GetSetKeyValues (wsSubClassRegKey, usbPhidKeyValues, SET, TRUE );
	if (!bSuccess)
	{
        DEBUGMSG (ZONE_ERROR, (TEXT("GetSetKeyValues failed!\n")));
		FreeLibrary (hUSBD);
		return FALSE;
    }

	FreeLibrary (hUSBD);
    DEBUGMSG (ZONE_USBLOAD, (DTAG TEXT("USBInstallDriver--\r\n")));
	return TRUE;
}

//======================================================================
// USBUnInstallDriver - Called to have the USB client clean up its
// registry entries.
// 
BOOL USBUnInstallDriver()
{
    WCHAR wsUsbDeviceID[] = CLASS_NAME_SZ;
    USB_DRIVER_SETTINGS usbDriverSettings = { DRIVER_SETTINGS };
	LPUN_REGISTER_CLIENT_SETTINGS lpfnUnRegisterClientSettings;
	BOOL bSuccess = FALSE;

    DEBUGMSG (ZONE_USBLOAD, (DTAG TEXT("USBUnInstallDriver++\r\n")));

	// Explicitly link to USBD DLL.  
	HINSTANCE hUSBD = LoadLibrary (TEXT("usbd.dll"));
	if (hUSBD == 0)
	{
	    DEBUGMSG (ZONE_ERROR, (DTAG TEXT("Can't load USBD.DLL\r\n")));
		return FALSE;
	}
	lpfnUnRegisterClientSettings = (LPUN_REGISTER_CLIENT_SETTINGS) GetProcAddress (hUSBD, TEXT("UnRegisterClientSettings"));
	// Clean up the registry
	int i;
	if (lpfnUnRegisterClientSettings)
	{
		for(i=PHID_PID_MIN;i<=PHID_PID_MAX;i++)
		{
			usbDriverSettings.dwVendorId = PHID_VID;
			usbDriverSettings.dwProductId = i;

			bSuccess = (lpfnUnRegisterClientSettings) (wsUsbDeviceID, NULL, &usbDriverSettings);
			if(!bSuccess) break;
		}
	}
	FreeLibrary (hUSBD);
    DEBUGMSG (ZONE_USBLOAD, (DTAG TEXT("USBUnInstallDriver--\r\n")));
	return bSuccess;
}

