//======================================================================
// Phidget - USB Phidget driver for Windows CE
//
// Author: Patrick McNeil
// Based On: WebCam.cpp by Douglas Boling
//======================================================================
#include "stdafx.h"
#include <windows.h>				// For all that Windows stuff
#include <winioctl.h>				// Needed for CTLCODE macro
#include <USBdi.h>					// USB includes
#include <usb100.h>					// USB includes

#include "phidgetsdk.h"				// IOCTL defs for driver
#include "phidget.h" 				// Local driver includes

//
// Globals
//
HINSTANCE hInst;					 // DLL instance handle

//
// Debug zone support
//
#ifdef DEBUG
// Debug zone structure
DBGPARAM dpCurSettings = 
{
	TEXT("Phidget"), 
	{
		TEXT("Errors"),TEXT("Warnings"),TEXT("Functions"), TEXT("Init"),
		TEXT("Driver Calls"),TEXT("USBLoad"),TEXT("Thread"),TEXT("Frame Info"), 
		TEXT("Packets"),TEXT("Timing"),TEXT("Read Data"),TEXT("Probe/Commit"),
		TEXT("Still Cap"),TEXT("Stream Frames"),TEXT("Feature Ctls"), TEXT("Undefined") 
	},
	0x9063
};
#endif //DEBUG

//======================================================================
// DllMain - DLL initialization entry point
//
BOOL WINAPI DllMain (HANDLE hinstDLL, DWORD dwReason, 
					 LPVOID lpvReserved) {
	hInst = (HINSTANCE)hinstDLL;

	switch (dwReason) {
		case DLL_PROCESS_ATTACH:
			DEBUGREGISTER(hInst);
			// Improve performance by passing on thread attach calls
			DisableThreadLibraryCalls (hInst);
		break;
	
		case DLL_PROCESS_DETACH:
			DEBUGMSG(ZONE_INIT, (DTAG TEXT("DLL_PROCESS_DETACH\r\n")));
			break;
	}
	return TRUE;
}
//======================================================================
// PHD_Init - Driver initialization function
//
DWORD PHD_Init (DWORD dwContext, LPCVOID lpvBusContext) {
	PDRVCONTEXT pDrv;

	// Allocate a driver instance structure.
	pDrv = GetConfigData (dwContext);
	if (pDrv)
	{
		if (pdd_DrvInit (pDrv))
		{
			// If failure, the drv context structure will be freed by
			// USBDeviceAttach
			DEBUGMSG (ZONE_ERROR | ZONE_INIT, (DTAG TEXT("pdd_Init Failure!")));
			LeaveCriticalSection (&pDrv->csDCall);
			return 0;
		}
	}
	return (DWORD)pDrv;
}
//======================================================================
// PHD_Deinit - Driver de-initialization function
//
BOOL PHD_Deinit (DWORD dwContext) 
{
	PDRVCONTEXT pDrv = (PDRVCONTEXT) dwContext;

	pdd_DrvUninit (pDrv);
	// The drv context structure will be freed by 
	// USBDeviceNotificationCallback
	return TRUE;
}
//======================================================================
// PHD_Open - Called when driver opened
//
DWORD PHD_Open (DWORD dwContext, DWORD dwAccess, DWORD dwShare) 
{
	PDRVCONTEXT pDrv = (PDRVCONTEXT) dwContext;

	DEBUGMSG (ZONE_EXENTRY, 
			  (DTAG TEXT("PHD_Open++ dwContext: %x\r\n"), dwContext));
	// Verify that the context handle is valid.
	if (pDrv && (pDrv->dwSize != sizeof (DRVCONTEXT))) {
		return 0;
	}
	EnterCriticalSection (&pDrv->csDCall);

	if (!pDrv->nNumOpens)
	{
		pdd_DrvOpen (pDrv);
	}
	pDrv->nNumOpens++;
	
	LeaveCriticalSection (&pDrv->csDCall);

	return (DWORD)pDrv;
}
//======================================================================
// PHD_Close - Called when driver closed
//
BOOL PHD_Close (DWORD dwOpen) {
	PDRVCONTEXT pDrv = (PDRVCONTEXT) dwOpen;

	DEBUGMSG (ZONE_EXENTRY, 
			  (DTAG TEXT("PHD_Close++ dwOpen: %x\r\n"), dwOpen));

	EnterCriticalSection (&pDrv->csDCall);

	if (pDrv && (pDrv->dwSize != sizeof (DRVCONTEXT))) {
		DEBUGMSG (ZONE_FUNC | ZONE_ERROR, 
				  (DTAG TEXT("PHD_Close failed\r\n")));
		SetLastError(ERROR_INVALID_HANDLE);
		return 0;
	}

	if (pDrv->nNumOpens)
		pDrv->nNumOpens--;

	// Notify the PDD that the app is closing the driver.  The 
	// PDD will stop any running stream.
	if (!pDrv->nNumOpens)
		pdd_DrvClose (pDrv);

	LeaveCriticalSection (&pDrv->csDCall);

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("PHD_Close--\r\n")));
	return TRUE;
}
//======================================================================
// PHD_Read - Called when driver read
// TODO: we need to handle this...
DWORD PHD_Read (DWORD dwOpen, LPVOID pBuffer, DWORD dwCount) {
	DWORD dwBytesRead = 0;
	PDRVCONTEXT pDrv;
	DWORD dwErr = ERROR_SUCCESS;

	DEBUGMSG (ZONE_EXENTRY, 
			  (DTAG TEXT("PHD_Read++ dwOpen: %x\r\n"), dwOpen));

	pDrv = (PDRVCONTEXT) dwOpen;
	EnterCriticalSection (&pDrv->csRead);

	dwErr = pdd_Read (pDrv, pBuffer, dwCount, &dwBytesRead);
	SetLastError (dwErr);

	LeaveCriticalSection (&pDrv->csRead);

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("PHD_Read--\r\n")));
	return dwErr ? -1 : dwBytesRead;
}
//======================================================================
// PHD_Write - Called when driver written
// TODO: we need to handle this...
DWORD PHD_Write (DWORD dwOpen, LPVOID pBuffer, DWORD dwCount) {
	DWORD dwBytesWritten = 0;
	PDRVCONTEXT pDrv;
	DWORD dwErr = ERROR_SUCCESS;

	DEBUGMSG (ZONE_EXENTRY, 
			  (DTAG TEXT("PHD_Write++ dwOpen: %x\r\n"), dwOpen));

	pDrv = (PDRVCONTEXT) dwOpen;
	EnterCriticalSection (&pDrv->csWrite);

	dwErr = pdd_Write (pDrv, pBuffer, dwCount, &dwBytesWritten);
	SetLastError (dwErr);

	LeaveCriticalSection (&pDrv->csWrite);

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("PHD_Write--\r\n")));
	return dwErr ? -1 : dwBytesWritten;
}
//======================================================================
// PHD_Seek - Called when SetFilePtr called
//
DWORD PHD_Seek (DWORD dwOpen, long lDelta, WORD wType) {
	DEBUGMSG (ZONE_EXENTRY,(DTAG TEXT("PHD_Seek++ dwOpen:%x %d %d\r\n"), 
			  dwOpen, lDelta, wType));

	DEBUGMSG (ZONE_EXENTRY, (DTAG TEXT("PHD_Seek--\r\n")));
	return 0;
}
//======================================================================
// PHD_IOControl - Called when DeviceIOControl called
// 
DWORD PHD_IOControl (DWORD dwOpen, DWORD dwCode, PBYTE pIn, DWORD dwIn,
					 PBYTE pOut, DWORD dwOut, DWORD *pdwBytesWritten) {
	PDRVCONTEXT pDrv;
	DWORD err = ERROR_INVALID_PARAMETER;

	DEBUGMSG (ZONE_EXENTRY, 
			  (DTAG TEXT("PHD_IOControl++ dwOpen: %x  dwCode: %x\r\n"),
			  dwOpen, dwCode));

	pDrv = (PDRVCONTEXT) dwOpen;
	EnterCriticalSection (&pDrv->csDCall);
	
	switch (dwCode) {
		case IOCTL_PHIDGET_DEVICE_GET_ATTRIBUTES:
			err = mdd_GetAttributes (pDrv, pOut, dwOut, pdwBytesWritten);
			break;
		case IOCTL_PHIDGET_DEVICE_GET_TAGSTRING:
			err = mdd_GetTagString (pDrv, pOut, dwOut, pdwBytesWritten);
			break;
		case IOCTL_PHIDGET_DEVICE_SET_TAGSTRING:
			err = mdd_SetTagString (pDrv, pIn, dwIn, pdwBytesWritten);
			break;
		case IOCTL_PHIDGET_DEVICE_GET_SERIALNUMBERSTRING:
			err = mdd_GetSerialNumberString (pDrv, pOut, dwOut, pdwBytesWritten);
			break;
		default:
			DEBUGMSG (ZONE_ERROR, 
			 (DTAG TEXT("PHD_IOControl: unknown code %x\r\n"), dwCode));
			return FALSE;
	}
	SetLastError (err);
	LeaveCriticalSection (&pDrv->csDCall);
	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("PHD_IOControl--\r\n")));
	return err ? FALSE : TRUE;
}
//======================================================================
// PHD_PowerDown - Called when system suspends
//
void PHD_PowerDown (DWORD dwContext) 
{
	return;
}
//======================================================================
// PHD_PowerUp - Called when resumes
//
void PHD_PowerUp (DWORD dwContext) 
{
	return;
}
//----------------------------------------------------------------------
// GetConfigData - Get the pointer to the driver context structure
//
PDRVCONTEXT GetConfigData (DWORD dwContext)
{
	int nLen, rc;
	DWORD dwLen, dwType, dwSize = 0;
	HKEY hKey;
	PDRVCONTEXT pDrv = 0;

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("GetConfigData++\r\n")));
	nLen = 0;
	// If ptr < 65K, it's a value, not a pointer.  
	if (dwContext < 0x10000) {
		return 0; 
	} else {
		__try {
			nLen = lstrlen ((LPTSTR)dwContext);
		}
		__except (EXCEPTION_EXECUTE_HANDLER) {
			nLen = 0;
		}
	}
	if (!nLen) {
		DEBUGMSG (ZONE_ERROR, (DTAG TEXT("dwContext not a ptr\r\n")));
		return 0;
	}

	// Open the Active key for the driver.
	rc = RegOpenKeyEx(HKEY_LOCAL_MACHINE,(LPTSTR)dwContext,0, 0, &hKey);

	if (rc == ERROR_SUCCESS) {
		// Read the key value.
		dwLen = sizeof(pDrv);
		rc = RegQueryValueEx (hKey, TEXT("ClientInfo"), NULL, &dwType,
								   (PBYTE)&pDrv, &dwLen);

		RegCloseKey(hKey);
		if ((rc == ERROR_SUCCESS) && (dwType == REG_DWORD))
		{
			__try {
				if (pDrv->dwSize != sizeof (DRVCONTEXT))
					pDrv = 0;
			}
			__except (EXCEPTION_EXECUTE_HANDLER) {
				pDrv = 0;
			}
		}
		else 
		{
			DEBUGMSG (ZONE_ERROR, (DTAG TEXT("Error reading registry\r\n")));
			pDrv = 0;
		}
	} else
		DEBUGMSG (ZONE_ERROR, (DTAG TEXT("Error opening Active key\r\n")));

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("GetConfigData--\r\n")));
	return pDrv;
}
//======================================================================
// MDD functions
//======================================================================
//---------------------------------------------------------------------------------------
// mdd_GetAttributes - Handles IOCTL_PHIDGET_DEVICE_GET_ATTRIBUTES Ioctl command
// 
int mdd_GetAttributes (PDRVCONTEXT pDrv, PBYTE pOut, DWORD dwOut, PDWORD pdwBytesWritten) 
{

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_GetAttributes++\r\n")));

	// Check parameters
	if (!pOut || (dwOut < sizeof (DWORD)))
		ERROR_INVALID_PARAMETER;
	int rc=0;

	__try 
	{
		memcpy (pOut, &pDrv->attr, sizeof (PHIDDEV_ATTRIBUTES));
		*pdwBytesWritten = sizeof (PHIDDEV_ATTRIBUTES);
	}
	__except (EXCEPTION_EXECUTE_HANDLER)
	{
		rc = ERROR_INVALID_PARAMETER;
	}

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_GetAttributes-- rc %d\r\n"),rc));
	return rc;

}

//---------------------------------------------------------------------------------------
// mdd_GetTagString - Handles IOCTL_PHIDGET_DEVICE_GET_TAGSTRING Ioctl command
//
int mdd_GetTagString (PDRVCONTEXT pDrv, PBYTE pOut, DWORD dwOut, 
					  PDWORD pdwBytesWritten) 
{

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_GetTagString++\r\n")));

	// Check parameters
	if (!pOut || (dwOut < sizeof (DWORD)))
		ERROR_INVALID_PARAMETER;

	PHIDDEV_STRING dwVal;
	int rc;

	rc = GetString (pDrv, 4, dwVal.string, 255, &(dwVal.size));
	if (rc == 0)
	{
		__try 
		{
			memcpy (pOut, &dwVal, sizeof (dwVal));
			*pdwBytesWritten = sizeof (dwVal);
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			rc = ERROR_INVALID_PARAMETER;
		}
	}
	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_GetTagString-- rc %d\r\n"),rc));
	return rc;
}

//---------------------------------------------------------------------------------------
// mdd_SetTagString - Handles IOCTL_PHIDGET_DEVICE_SET_TAGSTRING Ioctl command
//
int mdd_SetTagString (PDRVCONTEXT pDrv, PBYTE pIn, DWORD dwIn, PDWORD pdwBytesWritten) 
{
	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_SetTagString++\r\n")));

	// Check parameters
	if (!pIn || (dwIn < sizeof (PHIDDEV_STRING)) || !pdwBytesWritten)
		return ERROR_INVALID_PARAMETER;

	PPHIDDEV_STRING pdwVal = (PPHIDDEV_STRING)pIn;
	int rc;

	rc = pdd_SetTagString (pDrv, pdwVal, pdwBytesWritten);

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_SetTagString-- rc %d\r\n"),rc));
	return rc;
}

//---------------------------------------------------------------------------------------
// mdd_GetSerialNumberString - Handles IOCTL_PHIDGET_DEVICE_GET_SERIALNUMBERSTRING Ioctl command
//
int mdd_GetSerialNumberString (PDRVCONTEXT pDrv, PBYTE pOut, DWORD dwOut, 
					  PDWORD pdwBytesWritten) 
{

	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_GetSerialNumberString++\r\n")));

	// Check parameters
	if (!pOut || (dwOut < sizeof (DWORD)))
		ERROR_INVALID_PARAMETER;

	PHIDDEV_STRING dwVal;
	int rc;

	rc = GetString (pDrv, 3, dwVal.string, 255, &(dwVal.size));
	if (rc == 0)
	{
		__try 
		{
			memcpy (pOut, &dwVal, sizeof (dwVal));
			*pdwBytesWritten = sizeof (dwVal);
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			rc = ERROR_INVALID_PARAMETER;
		}
	}
	DEBUGMSG (ZONE_FUNC, (DTAG TEXT("mdd_GetSerialNumberString-- rc %d\r\n"),rc));
	return rc;
}

// Get an string from the device. Call with pszBuffer == NULL to get
// the character count required (then add 1 for the NULL terminator).
// This function is from the WindowsCE Hid Driver
DWORD
GetString(
    PDRVCONTEXT		pDrv,
    DWORD           dwIdx,     // Only used with stringType == HID_STRING_INDEXED 
    LPWSTR          pszBuffer, // Set to NULL to get character count
    DWORD           cchBuffer, // Count of chars that will fit into pszBuffer
                               // including the NULL terminator.
    PDWORD          pcchActual // Count of chars in the string NOT including 
                               // the NULL terminator
    )
{

    DWORD dwErr;

    if ( (pcchActual == NULL) || (dwIdx == 0) ) {
        DEBUGMSG(ZONE_ERROR, (_T("Invalid parameter\r\n")));
        dwErr = ERROR_INVALID_PARAMETER;
        goto EXIT;
    }

    if (pszBuffer != NULL) {
        DEBUGCHK(IsBadWritePtr(pszBuffer, cchBuffer * sizeof(*pszBuffer)) == FALSE);
    }
    DEBUGCHK(IsBadWritePtr(pcchActual, sizeof(*pcchActual)) == FALSE);

	dwErr = Pdd_GetString(pDrv, dwIdx, 
        pszBuffer, cchBuffer, pcchActual);

EXIT:
    return dwErr;
}