/* - Servo full -
 * This example simply displays the PhidgetServo information when it is attached and displays the servo 
 * positions with a slider for position manipulation
 *
 * Please note that this example was designed to work with only one Phidget Servo connected. 
 * For an example using multiple Phidget Servos, please see a "multiple" example in the Servos Examples folder.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets;  //for the servo class and the exceptions class
using Phidgets.Events; //for the event handling classes

namespace Servo_full
{
    public partial class Form1 : Form
    {
        private Servo servo; //Declare a servo object

        public Form1()
        {
            InitializeComponent();
        }

        //Initialization code
        private void Form1_Load(object sender, EventArgs e)
        {
            servo = new Servo();

            servo.Attach += new AttachEventHandler(servo_Attach);
            servo.Detach += new DetachEventHandler(servo_Detach);
            servo.Error += new ErrorEventHandler(servo_Error);

            servo.PositionChange += new PositionChangeEventHandler(servo_PositionChange);

            servoNumCombo.Enabled = false;
            positionTrk.SetRange(-22, 232); //the maximum bound of the phidget servo is 232, it's minimum bound is -23

            //This assumes that if there is a command line argument, it is a serial number
            //and we try to open that specific device. Otherwise, open any device.
            String[] args = Environment.GetCommandLineArgs();
            if (args.Length > 2 && args[2].Equals("remote"))
                servo.open(int.Parse(args[1]), null);
            else if (args.Length > 1)
                servo.open(int.Parse(args[1]));
            else
                servo.open();
        }

        //PhidgetServo attach event handling code
        void servo_Attach(object sender, AttachEventArgs e)
        {
            Servo attached = (Servo)sender;
            attachedTxt.Text = attached.Attached.ToString();
            nameTxt.Text = attached.Name;
            serialTxt.Text = attached.SerialNumber.ToString();
            versionTxt.Text = attached.Version.ToString();
            servosTxt.Text = attached.servos.Count.ToString();
            servoNumCombo.Enabled = true;

            for (int i = 0; i < attached.servos.Count; i++)
            {
                servoNumCombo.Items.Add(i);
            }

            servoNumCombo.SelectedIndex = 0;

            //Use a try-catch block around code where you are getting and displaying the servo position data
            //if the current position state has yet to be set, it will throw a PhidgetException for value not set
            //you can use this to test this and to display that the value is unknown
            try
            {
                positionTxt.Text = attached.servos[0].Position.ToString();
                positionTrk.Value = (int)attached.servos[0].Position;
                engagedCheckBox.Checked = servo.servos[0].Engaged;
            }
            catch (PhidgetException)
            {
                positionTxt.Text = "Unknown";
            }
        }

        //PhidgetServo detach event handling code
        void servo_Detach(object sender, DetachEventArgs e)
        {
            Servo detached = (Servo)sender;
            attachedTxt.Text = detached.Attached.ToString();
            nameTxt.Text = "";
            serialTxt.Text = "";
            versionTxt.Text = "";
            servosTxt.Text = "";
            servoNumCombo.Enabled = false;
            servoNumCombo.Items.Clear();
            positionTxt.Text = "";
            positionTrk.Value = 0;
            engagedCheckBox.Checked = false;
        }

        void servo_Error(object sender, ErrorEventArgs e)
        {
            Phidget phid = (Phidget)sender;
            DialogResult result;
            switch (e.Type)
            {
                case PhidgetException.ErrorType.PHIDGET_ERR_BADPASSWORD:
                    phid.close();
                    TextInputBox dialog = new TextInputBox("Error Event",
                        "Authentication error: This server requires a password.", "Please enter the password, or cancel.");
                    result = dialog.ShowDialog();
                    if (result == DialogResult.OK)
                        phid.open(int.Parse(Environment.GetCommandLineArgs()[1]), null, dialog.password);
                    else
                        Environment.Exit(0);
                    break;
                case PhidgetException.ErrorType.PHIDGET_ERR_NETWORK:
                    result = MessageBox.Show("Network Error: " + e.Description +
                        Environment.NewLine + "Keep trying to connect?", "Error Event", MessageBoxButtons.YesNo);
                    if (result == DialogResult.No)
                    {
                        phid.close();
                        Environment.Exit(0);
                    }
                    break;
                default:
                    MessageBox.Show(e.exception.Message, "Error Event");
                    break;
            }
        }

        //Servo position change event handling code
        void servo_PositionChange(object sender, PositionChangeEventArgs e)
        {
            Servo attached = (Servo)sender;
            if (servoNumCombo.SelectedIndex == e.Index)
            {
                try
                {
                    positionTxt.Text = attached.servos[e.Index].Position.ToString();
                    engagedCheckBox.Checked = attached.servos[e.Index].Engaged;
                }
                catch (PhidgetException)
                {
                    positionTxt.Text = "Unknown";
                }
            }
        }

        //Modify the servo position based on the value returned by the slider
        private void positionTrk_Scroll(object sender, EventArgs e)
        {
            int index = servoNumCombo.SelectedIndex;

            //A PhidgetException will be thrown if you try to set the position to any value NOT between -23 and 232
            try
            {
                servo.servos[index].Position = positionTrk.Value;
            }
            catch (PhidgetException)
            {
                MessageBox.Show("position value out of bounds!");
            }
            catch { }
        }
        
        //If using a 4-motor Phidget Servo, this code wi;; switch between the selected servos and display their position data
        private void servoNumCombo_SelectedIndexChanged(object sender, EventArgs e)
        {
            int index = servoNumCombo.SelectedIndex;

            //Use a try-catch block around code where you are getting and displaying the servo position data
            //if the current position state has yet to be set, it will throw a PhidgetException for value not set
            //you can use this to test this and to display that the value is unknown
            try
            {
                positionTxt.Text = servo.servos[index].Position.ToString();
                positionTrk.Value = (int)servo.servos[index].Position;
                engagedCheckBox.Checked = servo.servos[index].Engaged;
            }
            catch (PhidgetException)
            {
                positionTxt.Text = "Unknown";
                engagedCheckBox.Checked = false;
            }
            catch { }
        }

        //When the application is terminating, close the Phidget.
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            servo.Attach -= new AttachEventHandler(servo_Attach);
            servo.Detach -= new DetachEventHandler(servo_Detach);
            servo.PositionChange -= new PositionChangeEventHandler(servo_PositionChange);

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            servo.close();
        }

        private void engagedCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            int index = servoNumCombo.SelectedIndex;

            //A PhidgetException will be thrown if you try to set the position to any value NOT between -23 and 232
            try
            {
                servo.servos[index].Engaged = engagedCheckBox.Checked;
            }
            catch (PhidgetException)
            {
                MessageBox.Show("couldn't set engaged state");
            }
            catch { }
        }
    }
}