/* - PHSensor full -
 * This example displays the connected Phidget PHSensor device's details as well as the current 
 * sensor readings being generated.  This example also demonstrates the ability to modify the sensor's
 * sensitivity by setting it to a new double value in the provided textbox.
 * 
 * Please note that this example was designed to work with only one Phidget PHSensor connected. 
 * For an example showing how to use two Phidgets of the same time concurrently, please see the
 * Servo-multi example in the Servo Examples.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets; //Needed for the PHSensor class, Phidget base classes, and the PhidgetException class
using Phidgets.Events; //Needed for the Phidget event handler classes

namespace PHSensor_full
{
    public partial class Form1 : Form
    {
        private PHSensor ph_sensor; //Declare a PHSensor object

        public Form1()
        {
            InitializeComponent();
        }

        //Initialize our PHSensor object and hook the event handlers
        private void Form1_Load(object sender, EventArgs e)
        {
            sensitivityTxt.Enabled = false;

            ph_sensor = new PHSensor();

            //Hook the event handlers to the object
            ph_sensor.Attach += new AttachEventHandler(ph_sensor_Attach);
            ph_sensor.Detach += new DetachEventHandler(ph_sensor_Detach);
            ph_sensor.Error += new ErrorEventHandler(ph_sensor_Error);

            ph_sensor.PHChange += new PHChangeEventHandler(ph_sensor_PHChange);

            //This assumes that if there is a command line argument, it is a serial number
            //and we try to open that specific device. Otherwise, open any device.
            String[] args = Environment.GetCommandLineArgs();
            if (args.Length > 2 && args[2].Equals("remote"))
                ph_sensor.open(int.Parse(args[1]), null);
            else if (args.Length > 1)
                ph_sensor.open(int.Parse(args[1]));
            else
                ph_sensor.open();

        }

        //PHSensor Attach event handler...We'll populate the fields in the GUI and enable the modify sensitivity textbox
        void ph_sensor_Attach(object sender, AttachEventArgs e)
        {
            PHSensor attached = (PHSensor)sender;
            attachedTxt.Text = attached.Attached.ToString();
            nameTxt.Text = attached.Name;
            serialTxt.Text = attached.SerialNumber.ToString();
            versionTxt.Text = attached.Version.ToString();

            try
            {
                phTxt.Text = attached.sensor.PH.ToString();
            }
            catch (PhidgetException)
            {
                phTxt.Text = "Unknown";
            }

            try
            {
                potentialTxt.Text = attached.sensor.Potential.ToString();
            }
            catch (PhidgetException)
            {
                potentialTxt.Text = "Unknown";
            }

            try
            {
                sensitivityTxt.Text = attached.sensor.Sensitivity.ToString();
            }
            catch (PhidgetException)
            {
                sensitivityTxt.Text = "Unknown";
            }
            potentialRange.Text = "(" + ph_sensor.sensor.PotentialMin.ToString() + "mV - " +
                ph_sensor.sensor.PotentialMax.ToString() + "mV)";
            phRange.Text = "(" + ph_sensor.sensor.PHMin.ToString() + " - " +
                ph_sensor.sensor.PHMax.ToString() + ")";
            sensitivityTxt.Enabled = true;
            temperatureTxt.Enabled = true;
        }

        //PHSensor Detach event handler....Clear the fields and disable the modify sensitivity textbox so that
        //new sensity value can't be sent while there is no device attached, otherwise this would generate a PhidgetException
        void ph_sensor_Detach(object sender, DetachEventArgs e)
        {
            PHSensor detached = (PHSensor)sender;
            attachedTxt.Text = detached.Attached.ToString();
            nameTxt.Clear();
            serialTxt.Clear();
            versionTxt.Clear();

            phTxt.Clear();
            potentialTxt.Clear();
            sensitivityTxt.Enabled = false;
            temperatureTxt.Clear();
            temperatureTxt.Enabled = false;
        }

        //Error event handler..We'll simply display the error description in a popup messagebox window
        void ph_sensor_Error(object sender, ErrorEventArgs e)
        {
            Phidget phid = (Phidget)sender;
            DialogResult result;
            switch (e.Type)
            {
                case PhidgetException.ErrorType.PHIDGET_ERR_BADPASSWORD:
                    phid.close();
                    TextInputBox dialog = new TextInputBox("Error Event",
                        "Authentication error: This server requires a password.", "Please enter the password, or cancel.");
                    result = dialog.ShowDialog();
                    if (result == DialogResult.OK)
                        phid.open(int.Parse(Environment.GetCommandLineArgs()[1]), null, dialog.password);
                    else
                        Environment.Exit(0);
                    break;
                case PhidgetException.ErrorType.PHIDGET_ERR_NETWORK:
                    result = MessageBox.Show("Network Error: " + e.Description +
                        Environment.NewLine + "Keep trying to connect?", "Error Event", MessageBoxButtons.YesNo);
                    if (result == DialogResult.No)
                    {
                        phid.close();
                        Environment.Exit(0);
                    }
                    break;
                default:
                    MessageBox.Show(e.exception.Message, "Error Event");
                    break;
            }
        }

        //PH Change event handler...We'll display the current PH reading from the sensor to the
        //specified textbox as they come in
        void ph_sensor_PHChange(object sender, PHChangeEventArgs e)
        {
            try
            {
                phTxt.Text = e.PH.ToString();
                potentialTxt.Text = ph_sensor.sensor.Potential.ToString();
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.Description);
            }
        }

        //When a new double value is entered into the textbox, we will update the sensitivity value in the sensor
        //If no PH sensor is attached when we try to update the sensitivity value, it will throw a PhidgetException,
        //so we will catch it and deal with it accordingly
        //Also, the sensitivity is a double value, so we should prevent and other values to be sent.  Since I am using the
        //Double.parse method, it will throw a FormatException if anything other than a double value is entered, so
        //we'll catch that and deal with it accordingly
        private void sensitivityTxt_TextChanged(object sender, EventArgs e)
        {
            try
            {
                ph_sensor.sensor.Sensitivity = Double.Parse(sensitivityTxt.Text);
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.Description);
            }
            catch (FormatException)
            {
                MessageBox.Show("Invalid sensitivity format entered - must be a decimal number");
            }
        }

        //When the application is terminating, close the Phidget.
        private void Form1_FormClosed(object sender, FormClosedEventArgs e)
        {
            ph_sensor.Attach -= new AttachEventHandler(ph_sensor_Attach);
            ph_sensor.Detach -= new DetachEventHandler(ph_sensor_Detach);
            ph_sensor.Error -= new ErrorEventHandler(ph_sensor_Error);

            ph_sensor.PHChange -= new PHChangeEventHandler(ph_sensor_PHChange);

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            ph_sensor.close();

            ph_sensor = null;
        }

        private void temperatureTxt_TextChanged(object sender, EventArgs e)
        {
            Double temp;
            if(Double.TryParse(temperatureTxt.Text, out temp))
            {
                ph_sensor.sensor.Temperature = temp;
                phRange.Text = "(" + ph_sensor.sensor.PHMin.ToString() + " - " +
                    ph_sensor.sensor.PHMax.ToString() + ")";
            }
        }
    }
}