/* - MotorControl full -
 * This example demonstrates thebasic usage of a Phidget MotorControl.  It demonstrates the manipulation of the velocity and
 * acceleration of the motor, as well as displays the data from generated events.
 *
 * Please note that this example was designed to work with only one Phidget MotorControl connected. 
 * For an example showing how to use two Phidgets of the same time concurrently, please see the
 * Servo-multi example in the Servo Examples.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets; //Needed for the MotorControl class, Phidget base classes, and the PhidgetException class
using Phidgets.Events; //Needed for the Phidget event handling classes
using System.Collections;

namespace MotorControl_full
{
    public partial class Form1 : Form
    {
        private MotorControl motoControl; //Declare a MotorControl object
        private ArrayList inputArray;

        public Form1()
        {
            InitializeComponent();
        }

        //initialize the MotorConrol object and hook the event handlers
        private void Form1_Load(object sender, EventArgs e)
        {
            inputArray = new ArrayList();
            inputArray.Add(input0Chk);
            inputArray.Add(input1Chk);
            inputArray.Add(input2Chk);
            inputArray.Add(input3Chk);

            
            for (int i = 0; i < 4; i++)
            {
                ((CheckBox)inputArray[i]).Visible = false;
                ((CheckBox)inputArray[i]).Enabled = false;

            }

            velocityTrk.SetRange(-100, 100); //The minimum velocity for a motor is -100, the maximum is 100

            motoControl = new MotorControl();

            motoControl.Attach += new AttachEventHandler(motoControl_Attach);
            motoControl.Detach += new DetachEventHandler(motoControl_Detach);
            motoControl.Error += new ErrorEventHandler(motoControl_Error);

            motoControl.CurrentChange += new CurrentChangeEventHandler(motoControl_CurrentChange);
            motoControl.InputChange += new InputChangeEventHandler(motoControl_InputChange);
            motoControl.VelocityChange += new VelocityChangeEventHandler(motoControl_VelocityChange);

            //This assumes that if there is a command line argument, it is a serial number
            //and we try to open that specific device. Otherwise, open any device.
            String[] args = Environment.GetCommandLineArgs();
            if (args.Length > 2 && args[2].Equals("remote"))
                motoControl.open(int.Parse(args[1]), null);
            else if (args.Length > 1)
                motoControl.open(int.Parse(args[1]));
            else
                motoControl.open();
        }

        //MotorControl Attach event handler...populate the fields and controls
        void motoControl_Attach(object sender, AttachEventArgs e)
        {
            MotorControl attached = (MotorControl)sender;
            attachedTxt.Text = attached.Attached.ToString();
            nameTxt.Text = attached.Name;
            serialTxt.Text = attached.SerialNumber.ToString();
            versiontxt.Text = attached.Version.ToString();
            numMotorsTxt.Text = attached.motors.Count.ToString();
            numInputsTxt.Text = attached.inputs.Count.ToString();

            accelTrk.SetRange((int)attached.motors[0].AccelerationMin + 1, (int)attached.motors[0].AccelerationMax); //The minimum acceleration for a motor is 0, the maximum is 100
            accelTrk.Value = accelTrk.Minimum;

            int i;
            for (i = 0; i < attached.inputs.Count; i++)
            {
                ((CheckBox)inputArray[i]).Visible = true;
                ((CheckBox)inputArray[i]).Enabled = true;
            }

            for (i = 0; i < attached.motors.Count; i++)
            {
                motorCmb.Items.Add(i);
            }

            if (attached.Name == "Phidget Low Voltage Motor Controller 2-motor 4-input")
            {
                cChangeTxt.Clear();
                cChangeTxt.Visible = false;
                label12.Visible = false;
            }
            else
            {
                cChangeTxt.Clear();
                cChangeTxt.Visible = true;
                label12.Visible = true;
            }

            motorCmb.SelectedIndex = 0;
        }

        //MotorControl Detach event handler...Clear all the fields and disable all the controls
        void motoControl_Detach(object sender, DetachEventArgs e)
        {
            MotorControl detached = (MotorControl)sender;
            attachedTxt.Text = detached.Attached.ToString();
            nameTxt.Clear();
            serialTxt.Clear();
            versiontxt.Clear();
            numMotorsTxt.Clear();
            numInputsTxt.Clear();

            int i;
            for (i = 0; i < 4; i++)
            {
                ((CheckBox)inputArray[i]).Visible = false;
                ((CheckBox)inputArray[i]).Enabled = false;
            }

            motorCmb.SelectedIndex = 0;
            motorCmb.Items.Clear();
            motorCmb.Enabled = false;

            cChangeTxt.Clear();
            cChangeTxt.Visible = false;
            label12.Visible = false;
        }

        //error handler...display the error description in a messagebox
        void motoControl_Error(object sender, ErrorEventArgs e)
        {
            Phidget phid = (Phidget)sender;
            DialogResult result;
            switch (e.Type)
            {
                case PhidgetException.ErrorType.PHIDGET_ERR_BADPASSWORD:
                    phid.close();
                    TextInputBox dialog = new TextInputBox("Error Event",
                        "Authentication error: This server requires a password.", "Please enter the password, or cancel.");
                    result = dialog.ShowDialog();
                    if (result == DialogResult.OK)
                        phid.open(int.Parse(Environment.GetCommandLineArgs()[1]), null, dialog.password);
                    else
                        Environment.Exit(0);
                    break;
                case PhidgetException.ErrorType.PHIDGET_ERR_NETWORK:
                    result = MessageBox.Show("Network Error: " + e.Description +
                        Environment.NewLine + "Keep trying to connect?", "Error Event", MessageBoxButtons.YesNo);
                    if (result == DialogResult.No)
                    {
                        phid.close();
                        Environment.Exit(0);
                    }
                    break;
                default:
                    MessageBox.Show(e.exception.Message, "Error Event");
                    break;
            }
        }

        //Current Change listener...display the data from a current change event
        void motoControl_CurrentChange(object sender, CurrentChangeEventArgs e)
        {
            if (e.Index == (int)motorCmb.SelectedItem)
            {
                cChangeTxt.Text = e.Current.ToString();
            }
        }

        //Input change event handler...display the result of a digital input change
        void motoControl_InputChange(object sender, InputChangeEventArgs e)
        {
            ((CheckBox)inputArray[e.Index]).Checked = e.Value;
        }

        //Velocity Change event handler...Display the velocity data after a Velocity change
        void motoControl_VelocityChange(object sender, VelocityChangeEventArgs e)
        {
            if (e.Index == (int)motorCmb.SelectedItem)
            {
                vCurrentTxt.Text = motoControl.motors[e.Index].Velocity.ToString();
                //vCurrentTxt.Text = e.Velocity.ToString();
            }
        }

        //When a motor is selected, display its current velocity and acceleration data as well as the trakbar controls
        private void motorCmb_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                velocityTrk.Value = (int)motoControl.motors[(int)motorCmb.SelectedItem].Velocity;
            }
            catch (PhidgetException)
            {
                velocityTrk.Value = 0;
            }
            try
            {
                accelTrk.Value = (int)motoControl.motors[(int)motorCmb.SelectedItem].Acceleration;
            }
            catch (PhidgetException)
            {
                accelTrk.Value = accelTrk.Minimum;
            }
            vMaxTxt.Text = velocityTrk.Value.ToString();
            vCurrentTxt.Text = velocityTrk.Value.ToString();
            AccelTxt.Text = accelTrk.Value.ToString();
        }

        //When the acceleration trakbar is changed send this data to the motor control to affect the motor
        //If no motor control is attached, a PhidgetException will be thrown, so we will catch it and deal
        //with it accordingly
        private void accelTrk_Scroll(object sender, EventArgs e)
        {
            try
            {
                motoControl.motors[(int)motorCmb.SelectedItem].Acceleration = accelTrk.Value;
                AccelTxt.Text = accelTrk.Value.ToString();
            }
            catch (PhidgetException)
            {
                MessageBox.Show("No PhidgetMotorControl Connected!");
                attachedTxt.Text = "false";
                nameTxt.Clear();
                serialTxt.Clear();
                versiontxt.Clear();
                numMotorsTxt.Clear();
                numInputsTxt.Clear();

                int i;
                for (i = 0; i < 4; i++)
                {
                    ((CheckBox)inputArray[i]).Visible = false;
                    ((CheckBox)inputArray[i]).Enabled = false;
                }

                motorCmb.SelectedIndex = 0;
                motorCmb.Items.Clear();
                motorCmb.Enabled = false;
            }
        }

        //When the velocity trakbar is changed send this data tothe motor control to affect the motor
        //If no motor control is attached, a PhidgetException will be thrown, so we will catch it and deal
        //with it accordingly
        private void velocityTrk_Scroll(object sender, EventArgs e)
        {
            
            try
            {
                vMaxTxt.Text = velocityTrk.Value.ToString();
                motoControl.motors[(int)motorCmb.SelectedItem].Velocity = velocityTrk.Value;
            }
            catch (PhidgetException)
            {
                MessageBox.Show("No PhidgetMotorControl Connected!");
                attachedTxt.Text = "false";
                nameTxt.Clear();
                serialTxt.Clear();
                versiontxt.Clear();
                numMotorsTxt.Clear();
                numInputsTxt.Clear();

                int i;
                for (i = 0; i < 4; i++)
                {
                    ((CheckBox)inputArray[i]).Visible = false;
                    ((CheckBox)inputArray[i]).Enabled = false;
                }

                motorCmb.SelectedIndex = 0;
                motorCmb.Items.Clear();
                motorCmb.Enabled = false;
            }
        }

        //When the form is being close, make sure to stop all the motors and close the Phidget.
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            motoControl.Attach -= new AttachEventHandler(motoControl_Attach);
            motoControl.Detach -= new DetachEventHandler(motoControl_Detach);
            motoControl.Error -= new ErrorEventHandler(motoControl_Error);

            motoControl.InputChange -= new InputChangeEventHandler(motoControl_InputChange);
            motoControl.CurrentChange -= new CurrentChangeEventHandler(motoControl_CurrentChange);
            motoControl.VelocityChange -= new VelocityChangeEventHandler(motoControl_VelocityChange);

            if (motoControl.Attached)
            {
                foreach (MotorControlMotor motor in motoControl.motors)
                {
                    motor.Velocity = 0;
                }
            }

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            motoControl.close();

            motoControl = null;
        }
    }
}