/* - LED full -
 * This example demonstrates the key functionalities of a Phidget LED.  It will allow you to choose an LED 
 * index in order to control the LED plugged into that index port.  It will also display the phidget 
 * information for the connected LED.
 *
 * Please note that this example was designed to work with only one Phidget LED connected. 
 * For an example showing how to use two Phidgets of the same time concurrently, please see the
 * Servo-multi example in the Servo Examples.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets; //Needed for the LED class, phidget base classes, and the PhidgetException class
using Phidgets.Events; // Needed for the event handling classes

namespace LED_full
{
    public partial class Form1 : Form
    {
        //Our LED object that will be used to interface with a connected Phidget LED board
        private LED led;

        public Form1()
        {
            InitializeComponent();
        }

        //Create and initialize an LED object to control an attached LED.  Hook the event handlers to the object and open.
        private void Form1_Load(object sender, EventArgs e)
        {
            ledIndexCmb.Enabled = false;
            ledTrk.Enabled = false;
            ledTrk.SetRange(0, 100);
            
            led = new LED();

            led.Attach += new AttachEventHandler(led_Attach);
            led.Detach += new DetachEventHandler(led_Detach);
            led.Error += new ErrorEventHandler(led_Error);

            //This assumes that if there is a command line argument, it is a serial number
            //and we try to open that specific device. Otherwise, open any device.
            String[] args = Environment.GetCommandLineArgs();
            if (args.Length > 2 && args[2].Equals("remote"))
                led.open(int.Parse(args[1]), null);
            else if (args.Length > 1)
                led.open(int.Parse(args[1]));
            else
                led.open();
        }

        //LED Attach event handler...populate the information fields and initialize the index selector combo boxes 
        //according to the number of available LEDs on the connected Phidget LED board.  Enable all the editable controls.
        void led_Attach(object sender, AttachEventArgs e)
        {
            LED attached = (LED)sender;
            attachedTxt.Text = attached.Attached.ToString();
            nameTxt.Text = attached.Name;
            serialTxt.Text = attached.SerialNumber.ToString();
            versionTxt.Text = attached.Version.ToString();
            ledNumTxt.Text = attached.leds.Count.ToString();

            int i;
            for (i = 0; i < attached.leds.Count; i++)
            {
                ledIndexCmb.Items.Add(i);
            }
            ledIndexCmb.Enabled = true;
            ledIndexCmb.SelectedIndex = 0;
            ledTrk.Enabled = true;
            ledTrk.Value = 0;
        }

        //LED Detach event handler...we'll clear all our fields and disable all the editable fields to prevent errors
        void led_Detach(object sender, DetachEventArgs e)
        {
            LED detached = (LED)sender;
            attachedTxt.Text = detached.Attached.ToString();
            nameTxt.Text = "";
            serialTxt.Text = "";
            versionTxt.Text = "";
            ledNumTxt.Text = "";

            ledIndexCmb.Enabled = false;
            ledIndexCmb.Items.Clear();
            ledTrk.Enabled = false;
            ledTrk.Value = 0;
        }

        //Error event handler...We will just display the error message in a popup messagebox
        void led_Error(object sender, ErrorEventArgs e)
        {
            Phidget phid = (Phidget)sender;
            DialogResult result;
            switch (e.Type)
            {
                case PhidgetException.ErrorType.PHIDGET_ERR_BADPASSWORD:
                    phid.close();
                    TextInputBox dialog = new TextInputBox("Error Event",
                        "Authentication error: This server requires a password.", "Please enter the password, or cancel.");
                    result = dialog.ShowDialog();
                    if (result == DialogResult.OK)
                        phid.open(int.Parse(Environment.GetCommandLineArgs()[1]), null, dialog.password);
                    else
                        Environment.Exit(0);
                    break;
                case PhidgetException.ErrorType.PHIDGET_ERR_NETWORK:
                    result = MessageBox.Show("Network Error: " + e.Description +
                        Environment.NewLine + "Keep trying to connect?", "Error Event", MessageBoxButtons.YesNo);
                    if (result == DialogResult.No)
                    {
                        phid.close();
                        Environment.Exit(0);
                    }
                    break;
                default:
                    MessageBox.Show(e.exception.Message, "Error Event");
                    break;
            }
        }

        //When we select an LED index, in this case for the left bank of leds, set the trackbar to the selected 
        //LED's current brightness setting.  
        //A PhidgetException could be thrown when accessing the led collection from the phidget class if no LED 
        //phidget is connected, so we'llcatch it and deal with it accordingly
        private void ledIndexCmb_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                ledTrk.Value = led.leds[(int)ledIndexCmb.SelectedItem];
            }
            catch (PhidgetException)
            {
                //value is not yet known - that's ok
                ledTrk.Value = 0;
            }
        }

        //When we scroll the trackbar for a selected LED index, we want to set the LED to that value in the created object.
        //A PhidgetException may be thrown if there is no Phidget LED connected, so we'll catch it and deal with it
        //accordingly
        private void ledTrk_Scroll(object sender, EventArgs e)
        {
            try
            {
                led.leds[(int)ledIndexCmb.SelectedItem] = ledTrk.Value;
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show("Problem setting LED: "+ex.Message);
            }
        }

        //When the application is terminating, close the phidget
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            led.Attach -= new AttachEventHandler(led_Attach);
            led.Detach -= new DetachEventHandler(led_Detach);
            led.Error -= new ErrorEventHandler(led_Error);

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            led.close();
        }
    }
}