/* - InterfaceKit full -
 * In this example we display the interface kit details including the number of inputs and outputs it allows,
 * and displays the values of the inputs and outputs.
 * Note: This example was developed and run with an LED in digital output 0 and an LED in digital output 7 to test
 * the simulated digital output.
 *
 * Please note that this example was designed to work with only one Phidget InterfaceKit connected. 
 * For an example showing how to use two Phidgets of the same time concurrently, please see the
 * Servo-multi example in the Servo Examples.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets;  //needed for the interfacekit class and the phidget exception class
using Phidgets.Events;  //needed for the event handling classes

namespace InterfaceKit_full
{
    public partial class Form1 : Form
    {
        private InterfaceKit ifKit;
        private CheckBox[] digiInArray = new CheckBox[16];
        private CheckBox[] digiOutArray = new CheckBox[16];
        private TextBox[] sensorInArray = new TextBox[8];

        public Form1()
        {
            InitializeComponent();
            //Phidget.enableLogging(Phidget.LogLevel.PHIDGET_LOG_VERBOSE, null);
        }

        //initialize the device
        private void Form1_Load(object sender, EventArgs e)
        {

            makeDigiInArray();
            makeDigiOutArray();
            makeSensorInArray();

            label8.Visible = false;
            inputTrk.Value = 0;
            inputTrk.Enabled = false;
            inputTrk.Visible = false;
            sensitivityTxt.Text = "";
            sensitivityTxt.Visible = false;

            ratioChk.Enabled = false;
            ratioChk.Checked = false;
            ratioChk.Visible = false;

            try
            {
                ifKit = new InterfaceKit();

                ifKit.Attach += new AttachEventHandler(ifKit_Attach);
                ifKit.Detach += new DetachEventHandler(ifKit_Detach);
                ifKit.Error += new ErrorEventHandler(ifKit_Error);

                ifKit.InputChange += new InputChangeEventHandler(ifKit_InputChange);
                ifKit.OutputChange += new OutputChangeEventHandler(ifKit_OutputChange);
                ifKit.SensorChange += new SensorChangeEventHandler(ifKit_SensorChange);

                //This assumes that if there is a command line argument, it is a serial number
                //and we try to open that specific device. Otherwise, open any device.
                String[] args = Environment.GetCommandLineArgs();
                if (args.Length > 2 && args[2].Equals("remote"))
                {
                    attachedTxt.Text = "Connecting...";
                    ifKit.open(int.Parse(args[1]), null);
                }
                else if (args.Length > 1)
                    ifKit.open(int.Parse(args[1]));
                else
                    ifKit.open();

            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.ToString());
            }

        }

        

        //attach event handler... here we'll display the interface kit details as well as determine how many output
        //and input fields to display as well as determine the range of values for the output simulator slider
        void ifKit_Attach(object sender, AttachEventArgs e)
        {
            InterfaceKit ifKit = (InterfaceKit)sender;
            attachedTxt.Text = ifKit.Attached.ToString();
            nameTxt.Text = ifKit.Name;
            serialTxt.Text = ifKit.SerialNumber.ToString();
            versionTxt.Text = ifKit.Version.ToString();
            digiInNumTxt.Text = ifKit.inputs.Count.ToString();
            digiOutNumTxt.Text = ifKit.outputs.Count.ToString();
            sensorInNumTxt.Text = ifKit.sensors.Count.ToString();

            int i;
            for (i = 0; i < ifKit.inputs.Count; i++)
            {
                digiInArray[i].Visible = true;
                digiInArray[i].ForeColor = Color.Wheat;
                ((Label)digitalInputsGroupBox.Controls["digitalInputLabel" + i.ToString()]).Visible = true;
            }

            for (i = 0; i < ifKit.outputs.Count; i++)
            {
                digiOutArray[i].Visible = true;
                digiOutArray[i].Checked = false;
                digiOutArray[i].Enabled = true;
                ((Label)digitalOutputsGroupBox.Controls["digitalOutputLabel" + i.ToString()]).Visible = true;
            }

            if (ifKit.sensors.Count > 0)
            {
                for (i = 0; i < ifKit.sensors.Count; i++)
                {
                    sensorInArray[i].Visible = true;
                    ((Label)analogInputsGroupBox.Controls["analogInputLabel" + i.ToString()]).Visible = true;
                }

                label8.Visible = true;
                sensitivityTxt.Visible = true;
                inputTrk.Visible = true;
                ratioChk.Visible = true;
            }
            else
            {
                label8.Visible = false;
                sensitivityTxt.Visible = false;
                inputTrk.Visible = false;
                ratioChk.Visible = false;
            }

            inputTrk.Enabled = true;
            try
            {
                if (ifKit.sensors.Count > 0)
                    inputTrk.Value = ifKit.sensors[0].Sensitivity;

                sensitivityTxt.Text = inputTrk.Value.ToString();
            }
            catch { }

            try
            {
                ratioChk.Enabled = true;
                ratioChk.Checked = ifKit.ratiometric;
            }
            catch(PhidgetException pe)
            {
                if (pe.Type == PhidgetException.ErrorType.PHIDGET_ERR_UNSUPPORTED)
                {
                    ratioChk.Enabled = false;
                    ratioChk.Visible = false;
                }
            }

        }

        //ifkit detach event handler... here we display the statu, which will be false as the device is not attached. 
        //We will also clear the display fields and hide the inputs and outputs.
        void ifKit_Detach(object sender, DetachEventArgs e)
        {
            InterfaceKit ifKit = (InterfaceKit)sender;
            attachedTxt.Text = ifKit.Attached.ToString();
            nameTxt.Text = "";
            serialTxt.Text = "";
            versionTxt.Text = "";
            digiInNumTxt.Text = "";
            digiOutNumTxt.Text = "";
            sensorInNumTxt.Text = "";

            int i;
            for (i = 0; i < 16; i++)
            {
                digiInArray[i].Visible = false;
                digiInArray[i].Checked = false;
                ((Label)digitalInputsGroupBox.Controls["digitalInputLabel" + i.ToString()]).Visible = false;
            }
            for (i = 0; i < 16; i++)
            {
                digiOutArray[i].Enabled = false;
                digiOutArray[i].Visible = false;
                ((Label)digitalOutputsGroupBox.Controls["digitalOutputLabel" + i.ToString()]).Visible = false;
            }
            for (i = 0; i < 8; i++)
            {
                sensorInArray[i].Visible = false;
                sensorInArray[i].Text = "";
                ((Label)analogInputsGroupBox.Controls["analogInputLabel" + i.ToString()]).Visible = false;
            }

            label8.Visible = false;
            sensitivityTxt.Visible = false;
            inputTrk.Visible = false;
            inputTrk.Value = 0;
            inputTrk.Enabled = false;
            sensitivityTxt.Text = "";

            ratioChk.Enabled = false;
            ratioChk.Checked = false;
            ratioChk.Visible = false;
        }

        //error event handler
        void ifKit_Error(object sender, ErrorEventArgs e)
        {
            Phidget phid = (Phidget)sender;
            DialogResult result;
            switch (e.Type)
            {
                case PhidgetException.ErrorType.PHIDGET_ERR_BADPASSWORD:
                    phid.close();
                    TextInputBox dialog = new TextInputBox("Error Event", 
                        "Authentication error: This server requires a password.", "Please enter the password, or cancel.");
                    result = dialog.ShowDialog();
                    if (result == DialogResult.OK)
                        phid.open(int.Parse(Environment.GetCommandLineArgs()[1]), null, dialog.password);
                    else
                        Environment.Exit(0);
                    break;
                case PhidgetException.ErrorType.PHIDGET_ERR_NETWORK:
                    result = MessageBox.Show("Network Error: " + e.Description +
                        Environment.NewLine + "Keep trying to connect?", "Error Event", MessageBoxButtons.YesNo);
                    if (result == DialogResult.No)
                    {
                        phid.close();
                        Environment.Exit(0);
                    }
                    break;
                default:
                    MessageBox.Show(e.exception.Message, "Error Event");
                    break;
            }
        }

        //digital input change event handler... here we check or uncheck the corresponding input checkbox based 
        //on the index of the digital input that generated the event
        void ifKit_InputChange(object sender, InputChangeEventArgs e)
        {
            digiInArray[e.Index].Checked = e.Value;
        }

        //digital output change event handler... here we check or uncheck the corresponding output checkbox
        //based on the index of the output that generated the event
        void ifKit_OutputChange(object sender, OutputChangeEventArgs e)
        {
            digiOutArray[e.Index].Checked = e.Value;
        }

        //sensor input change event handler...Set the textbox content based on the input index that is 
        //communicating with the interface kit
        void ifKit_SensorChange(object sender, SensorChangeEventArgs e)
        {
            sensorInArray[e.Index].Text = e.Value.ToString();
        }

        //Modify the digital ouputs...Please observe the properties window in the ofrm designer for the 
        //digital output checkboxes. Each of the output checkboxes CheckedChanged events point to this one 
        //event handler, I use the sender object (the checkbox triggering the event) to get the checkbox that 
        //is changing.  Also note that there is a "tag" property that is basically user defined data associated 
        //with the control.  I used this to store the output index that the checkbox is supposed to represent 
        //for use in the following code to easily index the output we wanted to change.  Hopefully this clarifies what
        //was done in this method.
        private void CheckedChanged(object sender, EventArgs e)
        {
            CheckBox outputChk = (CheckBox)sender;

            int outputIndex = int.Parse((string)outputChk.Tag);

            ifKit.outputs[outputIndex] = outputChk.Checked;
        }

        //Modify the sensitivity of the analog inputs. In other words, the amount that the inputs must change between 
        //sensorchange events.
        private void inputTrk_Scroll(object sender, EventArgs e)
        {
            try
            {
                for (int i = 0; i < ifKit.sensors.Count; i++)
                {
                    ifKit.sensors[i].Sensitivity = inputTrk.Value;
                }
                sensitivityTxt.Text = inputTrk.Value.ToString();
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.Description);
            }
        }

        private void ratioChk_CheckedChanged(object sender, EventArgs e)
        {
            try
            {
                ifKit.ratiometric = ratioChk.Checked;
            }
            catch (PhidgetException) { }
        }

        //When the application is terminating, close the Phidget.
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            ifKit.Attach -= new AttachEventHandler(ifKit_Attach);
            ifKit.Detach -= new DetachEventHandler(ifKit_Detach);
            ifKit.InputChange -= new InputChangeEventHandler(ifKit_InputChange);
            ifKit.OutputChange -= new OutputChangeEventHandler(ifKit_OutputChange);
            ifKit.SensorChange -= new SensorChangeEventHandler(ifKit_SensorChange);
            ifKit.Error -= new ErrorEventHandler(ifKit_Error);

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            ifKit.close();
        }

        //this method creates the digital input array that corresponds to the group of checkboxes
        //we are using to represent the state of the digital inputs on the interface kit
        private void makeDigiInArray()
        {
            for (int i = 0; i < 16; i++)
            {
                ((Label)digitalInputsGroupBox.Controls["digitalInputLabel" + i.ToString()]).Visible = false;
                digiInArray[i] = (CheckBox)digitalInputsGroupBox.Controls["checkBox" + (i + 1).ToString()];
                digiInArray[i].Visible = false;
            }
        }

        //this method creates the digital output array that corresponds to the group of checkboxes
        //we are using to represent the state of the digital outputs on the interface kit
        private void makeDigiOutArray()
        {
            for (int i = 0; i < 16; i++)
            {
                ((Label)digitalOutputsGroupBox.Controls["digitalOutputLabel" + i.ToString()]).Visible = false;
                digiOutArray[i] = (CheckBox)digitalOutputsGroupBox.Controls["checkBox" + (i + 17).ToString()];
                digiOutArray[i].Visible = false;
            }
        }

        //this method creates the analog input array (mst likelely connected to analog sensors) that corresponds 
        //to the group of textboxes which will be holding the current values of the analog inputs that are being 
        //sent to the interface kit
        private void makeSensorInArray()
        {
            for (int i = 0; i < 8; i++)
            {
                ((Label)analogInputsGroupBox.Controls["analogInputLabel" + i.ToString()]).Visible = false;
                sensorInArray[i] = (TextBox)analogInputsGroupBox.Controls["textBox" + (i + 1).ToString()];
                sensorInArray[i].Visible = false;
            }
        }
    }
}