/* - TemperatureSensor simple -
 ****************************************************************************************
 * This simple example simple opens a temperature sensor phidget and waits for one to be 
 * attached.  The program will then wait for user input to terminate.  While waiting it
 * will display the data generated by the events, such as the temperature change event 
 * which will display the currently measured temperature without sensitivity
 * modifications.  For a more detailed example where sensitivity can be adjusted and all 
 * the functionalities of  a temperature sensor are demonstrated, see 
 * TemperatureSensor-full example.
 * 
 * Please note that this example was designed to work with only one Phidget 
 * TemperatureSensor connected. For an example using multiple Phidget TemperatureSensors,
 * please see a "multiple" example in the TemperatureSensor Examples folder.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */
using System;
using System.Collections.Generic;
using System.Text;
//Needed for the TemperatureSensor class, Phidget base classes, 
//and the PhidgetException class
using Phidgets; 
//Needed for the Phidget event handler classes
using Phidgets.Events; 

namespace TemperatureSensor_simple
{
    class Program
    {
        static void Main(string[] args)
        {
            try
            {
                //Declare a TemperatureSensor object
                TemperatureSensor tempSensor = new TemperatureSensor(); 
                
                //hook the basic event handlers
                tempSensor.Attach += new AttachEventHandler(tempSensor_Attach);
                tempSensor.Detach += new DetachEventHandler(tempSensor_Detach);
                tempSensor.Error += new ErrorEventHandler(tempSensor_Error);

                //hook the phidget specific event handlers
                tempSensor.TemperatureChange += new TemperatureChangeEventHandler
                                                        (tempSensor_TemperatureChange);

                //open the phidget object to allow connections with phidget temperature
                //sensor devices
                tempSensor.open();

                //Make the program wait for a Phidget TemperatureSensor to be attached 
                //before trying to communicate with one. If we try and call the methods 
                //of the tempSensor object without a TemperatureSensor device attached,
                //a PhidgetException
                //will be thrown
                Console.WriteLine("Waiting for TemperatureSensor to be attached....");
                tempSensor.waitForAttachment();

                //Get the program to wait for user input to continue so that we can see
                //some events fire and console output to be displayed
                Console.WriteLine("Hit any key to end the program....");
                Console.Read();

                //Since user input was read, we'll terminate the program, so close the 
                //tempSensor object
                tempSensor.close();

                //null he object to clear it from memory
                tempSensor = null;

                //If no exceptions were thrown at this point, the program is ok to 
                //terminate
                Console.WriteLine("ok");
            }
            catch (PhidgetException ex)
            {
                Console.WriteLine(ex.Description);
            }
        }

        //Attach event handler...we'll display the serial number of the 
        //TemperatureSensor device that was attached to the console
        static void tempSensor_Attach(object sender, AttachEventArgs e)
        {
            Console.WriteLine("Temperature Sensor {0} attached!", 
                                    e.Device.SerialNumber.ToString());
        }

        //Detach event handler...we'll display the serial number of the 
        //TemperatureSensor device that was detached to the console
        static void tempSensor_Detach(object sender, DetachEventArgs e)
        {
            Console.WriteLine("Temperature Sensor {0} detached!", 
                                    e.Device.SerialNumber.ToString());
        }

        //Error event handler...we'll display the error description to the console
        static void tempSensor_Error(object sender, ErrorEventArgs e)
        {
            Console.WriteLine(e.Description);
        }

        //Temperature Changed event handler...we'll display the current measured 
        //temperature to the console.
        static void tempSensor_TemperatureChange(object sender, 
                                                    TemperatureChangeEventArgs e)
        {
            Console.WriteLine("Temperature change: {0}", e.Temperature);
        }
    }
}
