/* - Stepper full -
 * This example demonstrates the basic usage of a Phidget Stepper controller.  It demonstrates the manipulation 
 * of the velocity, goal position and acceleration of the stepper, as well as displays the data from generated 
 * events including current position.  If the connected Phidget STepper is Bipolar, the user will be shown 
 * Current values as well as being able to manipulate stepper torque.
 *
 * Please note that this example was designed to work with only one Phidget Stepper connected. 
 * For an example using multiple Phidget Stepper, please see a "multiple" example in the Stepper Examples folder.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets; //Needed for the Stepper class, Phidget base classes, and the PhidgetException class
using Phidgets.Events; //Needed for the Phidget event handling classes

namespace Stepper_full
{
    public partial class Form1 : Form
    {
        private Stepper stepper; //Declare a Stepper object
        private int[] goalArray; //array to hold the set goal values for each stepper
        private CheckBox[] inputArray = new CheckBox[4];

        public Form1()
        {
            InitializeComponent();
        }

        //initialize the stepper object and hook the event handlers
        private void Form1_Load(object sender, EventArgs e)
        {
            
            inputArray[0] = input0Chk;
            inputArray[1] = input1Chk;
            inputArray[2] = input2Chk;
            inputArray[3] = input3Chk;

            for (int i = 0; i < 4; i++)
            {
                inputArray[i].Visible = false;
            }

            velocityTrk.Enabled = false;
            accelTrk.Enabled = false;
            positionTrk.Enabled = false;
            torqueTrk.Enabled = false;

            torqueTrk.Visible = false;
            torqueTrk.Enabled = false;
            torqueTrk.Value = 0;

            TorqueTxt.Visible = false;
            cChangeTxt.Visible = false;

            label12.Visible = false;
            label15.Visible = false;
            label17.Visible = false;

            stepper = new Stepper();

            stepper.Attach += new AttachEventHandler(stepper_Attach);
            stepper.Detach += new DetachEventHandler(stepper_Detach);
            stepper.Error += new ErrorEventHandler(stepper_Error);

            stepper.CurrentChange += new CurrentChangeEventHandler(stepper_CurrentChange);
            stepper.PositionChange += new StepperPositionChangeEventHandler(stepper_PositionChange);
            stepper.VelocityChange += new VelocityChangeEventHandler(stepper_VelocityChange);
            stepper.InputChange += new InputChangeEventHandler(stepper_InputChange);

            //This assumes that if there is a command line argument, it is a serial number
            //and we try to open that specific device. Otherwise, open any device.
            String[] args = Environment.GetCommandLineArgs();
            if (args.Length > 2 && args[2].Equals("remote"))
                stepper.open(int.Parse(args[1]), null);
            else if (args.Length > 1)
                stepper.open(int.Parse(args[1]));
            else
                stepper.open();

        }

        //Stepper attach event handler...populate the available fields and controls
        void stepper_Attach(object sender, AttachEventArgs e)
        {
            Stepper attached = (Stepper)sender;

            goalArray = new int[attached.steppers.Count];

            attachedTxt.Text = attached.Attached.ToString();
            nameTxt.Text = attached.Name;
            serialTxt.Text = attached.SerialNumber.ToString();
            versiontxt.Text = attached.Version.ToString();
            numSteppersTxt.Text = attached.steppers.Count.ToString();

            positionTrk.Value = 0;
            accelTrk.Value = 0;
            velocityTrk.Value = 0;

            goalTxt.Text = positionTrk.Value.ToString();

            int i;
            for (i = 0; i < attached.steppers.Count; i++)
            {
               
                stepperCmb.Items.Add(i);
                goalArray[i] = 0;
                stepper.steppers[i].MaxVelocity = 0;
                stepper.steppers[i].Acceleration = 0;
                stepper.steppers[i].Position = 0;
                
            }

            stepperCmb.SelectedIndex = 0;

            //Only a Bipolar stepper has access to these features
            if (attached.Name.CompareTo("Phidget Bipolar Stepper Controller 1-motor") == 0)
            {
                stepper.steppers[0].Torque = 0;

                torqueTrk.Visible = true;
                torqueTrk.Enabled = true;
                torqueTrk.Value = 0;

                TorqueTxt.Visible = true;
                cChangeTxt.Visible = true;

                label12.Visible = true;
                label15.Visible = true;
                label17.Visible = true;

                stepperCmb.Enabled = false;

                for (i = 0; i < 4; i++)
                {
                    inputArray[i].Visible = true;
                }

                velocityTrk.SetRange(0, 1020);
                accelTrk.SetRange(0, 6300);
                positionTrk.SetRange(-20000, 20000); //Have arbitrarily chosen a range of -20000 - 20000, but the maximum range is closer to +-549000000000
                torqueTrk.SetRange(0, 100); // Range for the torque is 0 - 100

                velocityTrk.Enabled = true;
                accelTrk.Enabled = true;
                positionTrk.Enabled = true;

               
            }
            else
            {
                torqueTrk.Visible = false;
                torqueTrk.Enabled = false;
                torqueTrk.Value = 0;

                TorqueTxt.Visible = false;
                TorqueTxt.Text = torqueTrk.Value.ToString();
                cChangeTxt.Visible = false;

                label12.Visible = false;
                label15.Visible = false;
                label17.Visible = false;

                stepperCmb.Enabled = true;

                velocityTrk.SetRange(0, 255);
                accelTrk.SetRange(0, 3150);
                positionTrk.SetRange(-20000, 20000); //Have arbitrarily chosen a range of -20000 - 20000, but the maximum range is closer to +-549000000000

                velocityTrk.Enabled = true;
                accelTrk.Enabled = true;
                positionTrk.Enabled = true;
            }
        }

        //Stepper Detach event handler...Clear all the fields and disable all the controls
        void stepper_Detach(object sender, DetachEventArgs e)
        {
            Stepper detached = (Stepper)sender;
            attachedTxt.Text = detached.Attached.ToString();
            nameTxt.Clear();
            serialTxt.Clear();
            versiontxt.Clear();
            numSteppersTxt.Clear();

            stepperCmb.Items.Clear();
            stepperCmb.SelectedIndex = 0;
            stepperCmb.Enabled = false;

            positionTrk.Value = 0;
            positionTrk.Enabled = false;

            //Only a Bipolar stepper has access to these features
            if (detached.Name.CompareTo("Phidget Bipolar Stepper Controller 1-motor") == 0)
            {
                torqueTrk.Value = 0;
                TorqueTxt.Clear();
                cChangeTxt.Clear();
            }
            torqueTrk.Visible = false;
            torqueTrk.Enabled = false;
            TorqueTxt.Visible = false;
            cChangeTxt.Visible = false;

            label12.Visible = false;
            label15.Visible = false;
            label17.Visible = false;

            int i;
            for (i = 0; i < 4; i++)
            {
                inputArray[i].Visible = false;
            }

            velocityTrk.Enabled = false;
            accelTrk.Enabled = false;
            positionTrk.Enabled = false;
        }

        //Stepper Error event handler...Display the details of the error in a message box
        void stepper_Error(object sender, ErrorEventArgs e)
        {
            MessageBox.Show(e.Description);
        }

        //Stepper Current change event handler...Display the current data from the current change event
        void stepper_CurrentChange(object sender, CurrentChangeEventArgs e)
        {
            if (e.Index == (int)stepperCmb.SelectedItem)
            {
                cChangeTxt.Text = e.Current.ToString();
            }
        }

        //Stepper position change event handler...Display the position data from the position change event
        void stepper_PositionChange(object sender, StepperPositionChangeEventArgs e)
        {
            if (e.Index == (int)stepperCmb.SelectedItem)
            {
                positionTxt.Text = e.Position.ToString();
            }
        }

        //Stepper Velocity Change event handler...Display the data from the velocity change event
        void stepper_VelocityChange(object sender, VelocityChangeEventArgs e)
        {
            if (e.Index == (int)stepperCmb.SelectedItem)
            {
                velocityTxt.Text = e.Velocity.ToString();
                
            }
        }

        void stepper_InputChange(object sender, InputChangeEventArgs e)
        {

            inputArray[e.Index].Checked = e.Value;
        }

        //Selected index changed event handler...display the data from the newly selected stepper in the fields and setup the controls
        private void stepperCmb_SelectedIndexChanged(object sender, EventArgs e)
        {
            velocityTxt.Text = stepper.steppers[(int)stepperCmb.SelectedItem].Velocity.ToString();
            AccelTxt.Text = stepper.steppers[(int)stepperCmb.SelectedItem].Acceleration.ToString();
            positionTxt.Text = stepper.steppers[(int)stepperCmb.SelectedItem].Position.ToString();
            velocityTrk.Value = (int)stepper.steppers[(int)stepperCmb.SelectedItem].MaxVelocity;
            accelTrk.Value = (int)stepper.steppers[(int)stepperCmb.SelectedItem].Acceleration;
            positionTrk.Value = goalArray[(int)stepperCmb.SelectedItem];

            goalTxt.Text = goalArray[(int)stepperCmb.SelectedItem].ToString();
        }

        //Acceleration trackbar change event handler...update the acceleration value of the STepper and display the new value in the text box
        private void accelTrk_Scroll(object sender, EventArgs e)
        {
            try
            {
                stepper.steppers[(int)stepperCmb.SelectedItem].Acceleration = (double)accelTrk.Value;
                AccelTxt.Text = stepper.steppers[(int)stepperCmb.SelectedItem].Acceleration.ToString();
            }
            catch (PhidgetException)
            {
                MessageBox.Show("No Phidget Stepper controller connected!");
                attachedTxt.Text = "false";
                nameTxt.Clear();
                serialTxt.Clear();
                versiontxt.Clear();
                numSteppersTxt.Clear();

                stepperCmb.Items.Clear();
                stepperCmb.SelectedIndex = 0;
                stepperCmb.Enabled = false;

                positionTrk.Value = 0;
                positionTrk.Enabled = false;

                //Only a Bipolar stepper has access to these features
                if (stepper.Name.CompareTo("Phidget Bipolar Stepper Controller 1-motor") == 0)
                {
                    torqueTrk.Value = 0;
                    TorqueTxt.Clear();
                    cChangeTxt.Clear();

                    int i;
                    for (i = 0; i < 4; i++)
                    {
                        inputArray[i].Visible = false;
                    }
                }
                torqueTrk.Visible = false;
                torqueTrk.Enabled = false;
                TorqueTxt.Visible = false;
                cChangeTxt.Visible = false;

                label12.Visible = false;
                label15.Visible = false;
                label17.Visible = false;
            }
        }

        private void positionTrk_Scroll(object sender, EventArgs e)
        {
            try
            {
                stepper.steppers[(int)stepperCmb.SelectedItem].Position = positionTrk.Value;
                goalArray[(int)stepperCmb.SelectedItem] = positionTrk.Value;
                goalTxt.Text = goalArray[(int)stepperCmb.SelectedItem].ToString();
            }
            catch (PhidgetException)
            {
                MessageBox.Show("No Phidget Stepper controller connected!");
                attachedTxt.Text = "false";
                nameTxt.Clear();
                serialTxt.Clear();
                versiontxt.Clear();
                numSteppersTxt.Clear();

                stepperCmb.Items.Clear();
                stepperCmb.SelectedIndex = 0;
                stepperCmb.Enabled = false;

                positionTrk.Value = 0;
                positionTrk.Enabled = false;

                //Only a Bipolar stepper has access to these features
                if (stepper.Name.CompareTo("Phidget Bipolar Stepper Controller 1-motor") == 0)
                {
                    torqueTrk.Value = 0;
                    TorqueTxt.Clear();
                    cChangeTxt.Clear();

                    int i;
                    for (i = 0; i < 4; i++)
                    {
                        inputArray[i].Visible = false;
                    }
                }
                torqueTrk.Visible = false;
                torqueTrk.Enabled = false;
                TorqueTxt.Visible = false;
                cChangeTxt.Visible = false;

                label12.Visible = false;
                label15.Visible = false;
                label17.Visible = false;
            }
        }

        private void torqueTrk_Scroll(object sender, EventArgs e)
        {
            try
            {
                stepper.steppers[(int)stepperCmb.SelectedItem].Torque = torqueTrk.Value;
                TorqueTxt.Text = torqueTrk.Value.ToString();
            }
            catch (PhidgetException)
            {
                MessageBox.Show("No Phidget Stepper controller connected!");
                attachedTxt.Text = "false";
                nameTxt.Clear();
                serialTxt.Clear();
                versiontxt.Clear();
                numSteppersTxt.Clear();

                stepperCmb.Items.Clear();
                stepperCmb.SelectedIndex = 0;
                stepperCmb.Enabled = false;

                positionTrk.Value = 0;
                positionTrk.Enabled = false;

                //Only a Bipolar stepper has access to these features
                if (stepper.Name.CompareTo("Phidget Bipolar Stepper Controller 1-motor") == 0)
                {
                    torqueTrk.Value = 0;
                    TorqueTxt.Clear();
                    cChangeTxt.Clear();

                    int i;
                    for (i = 0; i < 4; i++)
                    {
                        inputArray[i].Visible = false;
                    }
                }
                torqueTrk.Visible = false;
                torqueTrk.Enabled = false;
                TorqueTxt.Visible = false;
                cChangeTxt.Visible = false;

                label12.Visible = false;
                label15.Visible = false;
                label17.Visible = false;
            }
        }

        private void velocityTrk_Scroll(object sender, EventArgs e)
        {
            try
            {
                stepper.steppers[(int)stepperCmb.SelectedItem].MaxVelocity = (double)velocityTrk.Value;
            }
            catch (PhidgetException)
            {
                MessageBox.Show("No Phidget Stepper controller connected!");
                attachedTxt.Text = "false";
                nameTxt.Clear();
                serialTxt.Clear();
                versiontxt.Clear();
                numSteppersTxt.Clear();

                stepperCmb.Items.Clear();
                stepperCmb.SelectedIndex = 0;
                stepperCmb.Enabled = false;

                positionTrk.Value = 0;
                positionTrk.Enabled = false;

                //Only a Bipolar stepper has access to these features
                if (stepper.Name.CompareTo("Phidget Bipolar Stepper Controller 1-motor") == 0)
                {
                    torqueTrk.Value = 0;
                    TorqueTxt.Clear();
                    cChangeTxt.Clear();

                    int i;
                    for (i = 0; i < 4; i++)
                    {
                        inputArray[i].Visible = false;
                    }
                }
                torqueTrk.Visible = false;
                torqueTrk.Enabled = false;
                TorqueTxt.Visible = false;
                cChangeTxt.Visible = false;

                label12.Visible = false;
                label15.Visible = false;
                label17.Visible = false;
            }
        }

        private void Form1_FormClosed(object sender, FormClosedEventArgs e)
        {
            stepper.Attach -= stepper_Attach;
            stepper.Detach -= stepper_Detach;
            stepper.Error -= stepper_Error;

            stepper.CurrentChange -=stepper_CurrentChange;
            stepper.PositionChange -= stepper_PositionChange;
            stepper.VelocityChange -= stepper_VelocityChange;

            for (int i = 0; i < stepper.steppers.Count; i++)
            {
                stepper.steppers[i].MaxVelocity = 0;
                stepper.steppers[i].Acceleration = 0;
                stepper.steppers[i].Position = 0;
                stepper.steppers[i].Torque = 0;
            }

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            stepper.close();

            stepper = null;
        }
    }
}