/* - Accelerometer full -
 * This example simply displays the accelerometer info if it is attached and displays the acceleration data 
 * for each axis as it is changing in pretty much raw form.  It also allows for modifying the sensitivity of 
 * each axis that is availabl on the attached accelerometer.

 * Please note that this example was designed to work with only one Phidget Accelerometer connected. For an 
 * example using multiple Phidget Accelerometers, please see a "multiple" example in the Accelerometer Examples folder.

 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */


using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets;  //needed for the accelerometer class and the phidgets exception class
using Phidgets.Events; //needed for the phidget event handling

namespace Accelerometer_full
{
    public partial class Form1 : Form
    {
        private Accelerometer accel;

        public Form1()
        {
            InitializeComponent();
        }

        //initalize the phidget device and link the event handler code
        private void Form1_Load(object sender, EventArgs e)
        {
            accel = new Accelerometer();

            accel.Attach += new AttachEventHandler(accel_Attach);
            accel.Detach += new DetachEventHandler(accel_Detach);
            accel.Error += new ErrorEventHandler(accel_Error);

            accel.AccelerationChange += new AccelerationChangeEventHandler(accel_AccelerationChange);

            //This assumes that if there is a command line argument, it is a serial number
            //and we try to open that specific device. Otherwise, open any device.
            String[] args = Environment.GetCommandLineArgs();
            if (args.Length > 2 && args[2].Equals("remote"))
                accel.open(int.Parse(args[1]), null);
            else if (args.Length > 1)
                accel.open(int.Parse(args[1]));
            else
                accel.open();
        }

        //accelerometer attach event handler
        void accel_Attach(object sender, AttachEventArgs e)
        {
            Accelerometer attached = (Accelerometer)sender;
            attachedTxt.Text = attached.Attached.ToString();
            nameTxt.Text = attached.Name;
            serialTxt.Text = attached.SerialNumber.ToString();
            versionTxt.Text = attached.Version.ToString();
            axesTxt.Text = attached.axes.Count.ToString();
            sensitivity1Txt.Enabled = true;
            sensitivity2Txt.Enabled = true;
            rangeTxt.Text = attached.axes[0].AccelerationMin == -attached.axes[0].AccelerationMax ?
                "" + attached.axes[0].AccelerationMax.ToString() + "g" :
                attached.axes[0].AccelerationMin.ToString() + "g to " + attached.axes[0].AccelerationMax.ToString() + "g";

            //depedning what kind of accelerometer you have (2-axis vs. 3-axis) enable or disable the axis-3 
            //sensitivity settings text box
            if (attached.axes.Count > 2)
            {
                sensitivity3Txt.Enabled = true;
            }
            else
            {
                sensitivity3Txt.Enabled = false;
            }

        }

        //accelerometer detach event handler
        void accel_Detach(object sender, DetachEventArgs e)
        {
            Accelerometer detached = (Accelerometer)sender;
            attachedTxt.Text = detached.Attached.ToString();
            nameTxt.Text = "";
            serialTxt.Text = "";
            versionTxt.Text = "";
            axesTxt.Text = "";
            sensitivity1Txt.Text = "";
            sensitivity1Txt.Enabled = false;
            sensitivity2Txt.Text = "";
            sensitivity2Txt.Enabled = false;
            sensitivity3Txt.Text = "";
            sensitivity3Txt.Enabled = false;
            acceleration1Txt.Text = "";
            acceleration2Txt.Text = "";
            acceleration3Txt.Text = "";
            rangeTxt.Text = "";
        }

        //accelerometer error event handler
        void accel_Error(object sender, ErrorEventArgs e)
        {
            MessageBox.Show(e.Description);
        }

        //acceleration change event handler
        void accel_AccelerationChange(object sender, AccelerationChangeEventArgs e)
        {
            Accelerometer attached = (Accelerometer)sender;

            try
            {
                switch (e.Index)
                {
                    case 0:
                        acceleration1Txt.Text = attached.axes[e.Index].Acceleration.ToString();
                        break;
                    case 1:
                        acceleration2Txt.Text = attached.axes[e.Index].Acceleration.ToString();
                        break;
                    case 2:
                        acceleration3Txt.Text = attached.axes[e.Index].Acceleration.ToString();
                        break;
                    default:
                        MessageBox.Show("Unexpected index provided");
                        break;
                }
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.ToString());
            }
        }

        //fire an event to modify the sensity on axis one based on what was entered into the corresponding text box
        private void sensitivity1Txt_TextChanged(object sender, EventArgs e)
        {
            try
            {
                accel.axes[0].Sensitivity = Double.Parse(sensitivity1Txt.Text);
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.ToString());
            }
            catch (FormatException)
            {
                //since I am parsing the string daa from the text boxes to doubles, the data that is entered there needs to be a double
                //if you empty the box it will through an exception, so default it to 0
                accel.axes[0].Sensitivity = 0;
                sensitivity1Txt.Text = "0";
            }
        }

        //fire an event to modify the sensity on axis two based on what was entered into the corresponding text box
        private void sensitivity2Txt_TextChanged(object sender, EventArgs e)
        {
            try
            {
                accel.axes[1].Sensitivity = Double.Parse(sensitivity2Txt.Text);
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.ToString());
            }
            catch (FormatException)
            {
                //since I am parsing the string daa from the text boxes to doubles, the data that is entered there needs to be a double
                //if you empty the box it will through an exception, so default it to 0
                accel.axes[1].Sensitivity = 0;
                sensitivity2Txt.Text = "0";
            }
        }

        //fire an event to modify the sensity on axis three based on what was entered into the corresponding text box
        private void sensitivity3Txt_TextChanged(object sender, EventArgs e)
        {
            try
            {
                accel.axes[2].Sensitivity = Double.Parse(sensitivity3Txt.Text);
            }
            catch (PhidgetException ex)
            {
                MessageBox.Show(ex.ToString());
            }
            catch (FormatException)
            {
                //since I am parsing the string daa from the text boxes to doubles, the data that is entered there needs to be a double
                //if you empty the box it will through an exception, so default it to 0
                accel.axes[2].Sensitivity = 0;
                sensitivity3Txt.Text = "0";
            }
        }

        //When the application is being terminated, close the Phidget
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            accel.Attach -= new AttachEventHandler(accel_Attach);
            accel.Detach -= new DetachEventHandler(accel_Detach);
            accel.AccelerationChange -= new AccelerationChangeEventHandler(accel_AccelerationChange);
            accel.Error -= new ErrorEventHandler(accel_Error);

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            accel.close();
        }
    }
}