/* - TemperatureSensor full -
 * This example displays the connected Phidget TemperatureSensor device's details as well as the current 
 * sensor readings being generated by the thermocouple.  This example also demonstrates the ability to
 * modify the sensor's sensitivity by setting it to a new double value in the provided textbox.
 * 
 * Please note that this example was designed to work with only one Phidget TemperatureSensor connected. 
 * For an example using multiple Phidget TemperatureSensors, please see a "multiple" example in the 
 * TemperatureSensor Examples folder.
 *
 * Copyright 2007 Phidgets Inc.  
 * This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using Phidgets; //Needed for the TemperatureSensor class, Phidget base classes, and the PhidgetException class
using Phidgets.Events; //Needed for the Phidget event handler classes

namespace TemperatureSensor_full
{
    public partial class Form1 : Form
    {
        private TemperatureSensor tempSensor; //Declare a TemperatureSensor object

        public Form1()
        {
            InitializeComponent();
        }

        //Initialize our temperatureSensor object and hook the event handlers
        private void Form1_Load(object sender, EventArgs e)
        {
            tempSensor = new TemperatureSensor();

            SensitivityTrk.SetRange(0, 200);
            SensitivityTrk.Enabled = false;

            //Hook the event handlers to the object
            tempSensor.Attach += new AttachEventHandler(tempSensor_Attach);
            tempSensor.Detach += new DetachEventHandler(tempSensor_Detach);
            tempSensor.Error += new ErrorEventHandler(tempSensor_Error);

            tempSensor.TemperatureChange += new TemperatureChangeEventHandler(tempSensor_TemperatureChange);

            //This assumes that if there is a command line argument, it is a serial number
            //and we try to open that specific device. Otherwise, open any device.
            String[] args = Environment.GetCommandLineArgs();
            if (args.Length > 2 && args[2].Equals("remote"))
                tempSensor.open(int.Parse(args[1]), null);
            else if (args.Length > 1)
                tempSensor.open(int.Parse(args[1]));
            else
                tempSensor.open();
        }

        //TemperatureSensor Attach event handler...We'll populate the fields in the GUI and enable the modify sensitivity textbox
        void tempSensor_Attach(object sender, AttachEventArgs e)
        {
            TemperatureSensor attached = (TemperatureSensor)sender;
            attachedTxt.Text = attached.Attached.ToString();
            nameTxt.Text = attached.Name;
            serialTxt.Text = attached.SerialNumber.ToString();
            versionTxt.Text = attached.Version.ToString();
            numThermoTxt.Text = attached.thermocouples.Count.ToString();
            SensitivityTrk.Enabled = true;
            SensitivityTrk.Value = (int)attached.thermocouples[0].Sensitivity*100;
            sensitivityTxt.Text = SensitivityTrk.Value.ToString();
            tempSensor.thermocouples[0].Sensitivity = (double)SensitivityTrk.Value / 100.0;
            onboardTempTxt.Text = attached.ambientSensor.Temperature.ToString();

            thermoRange.Text = "(" + tempSensor.thermocouples[0].TemperatureMin.ToString() + "C - " +
                tempSensor.thermocouples[0].TemperatureMax.ToString() + "C)";
            try
            {
                potentialRange.Text = "(" + tempSensor.thermocouples[0].PotentialMin.ToString() + "mV - " +
                    tempSensor.thermocouples[0].PotentialMax.ToString() + "mV)";
            }
            catch (PhidgetException) { potentialRange.Text = "(Not Supported)"; }
            ambientRange.Text = "(" + tempSensor.ambientSensor.TemperatureMin.ToString() + "C - " +
                tempSensor.ambientSensor.TemperatureMax.ToString() + "C)";

            thermoTypeComboBox.Enabled = true;
            switch (tempSensor.thermocouples[0].Type)
            {
                case TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_K_TYPE:
                    thermoTypeComboBox.SelectedItem = "K-Type";
                    break;
                case TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_J_TYPE:
                    thermoTypeComboBox.SelectedItem = "J-Type";
                    break;
                case TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_E_TYPE:
                    thermoTypeComboBox.SelectedItem = "E-Type";
                    break;
                case TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_T_TYPE:
                    thermoTypeComboBox.SelectedItem = "T-Type";
                    break;
            }
        }

        //TemperatureSensor Detach event handler....Clear the fields and disable the modify sensitivity textbox so that
        //new sensity value can't be sent while there is no device attached, otherwise this would generate a PhidgetException
        void tempSensor_Detach(object sender, DetachEventArgs e)
        {
            TemperatureSensor detached = (TemperatureSensor)sender;
            attachedTxt.Text = detached.Attached.ToString();
            nameTxt.Text = "";
            serialTxt.Text = "";
            versionTxt.Text = "";
            numThermoTxt.Text = "";
            thermoTempTxt.Text = "";
            thermoPotTxt.Text = "";
            onboardTempTxt.Text = "";
            SensitivityTrk.Enabled = false;
            SensitivityTrk.Value = 0;
            sensitivityTxt.Clear();
            thermoRange.Text = "";
            potentialRange.Text = "";
            ambientRange.Text = "";
            thermoTypeComboBox.Enabled = false;
        }

        //Error event handler..We'll simply display the error description in a popup messagebox window
        void tempSensor_Error(object sender, ErrorEventArgs e)
        {
            MessageBox.Show(e.Description);
        }

        //Temperature Change event handler...We'll display the current temperature reading from the sensor to the
        //specified textbox as they come in
        void tempSensor_TemperatureChange(object sender, TemperatureChangeEventArgs e)
        {
            thermoTempTxt.Text = e.Temperature.ToString();
            thermoPotTxt.Text = e.Potential.ToString();
            onboardTempTxt.Text = ((TemperatureSensor)sender).ambientSensor.Temperature.ToString();
        }

        //When a new double valu is entered into the textbox, we will update the sensitivity value in the sensor
        //If no temperature sensor is attached when we try to update the sensitivity value, it will throw a PhidgetException,
        //so we will catch it and deal with it accordingly
        private void SensitivityTrk_Scroll(object sender, EventArgs e)
        {
            tempSensor.thermocouples[0].Sensitivity = (double)SensitivityTrk.Value/100.0;
            sensitivityTxt.Text = (SensitivityTrk.Value/100.0).ToString();
        }

        //When the application is terminating, close the Phidget.
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            tempSensor.Attach -= new AttachEventHandler(tempSensor_Attach);
            tempSensor.Detach -= new DetachEventHandler(tempSensor_Detach);
            tempSensor.Error -= new ErrorEventHandler(tempSensor_Error);
            tempSensor.TemperatureChange -= new TemperatureChangeEventHandler(tempSensor_TemperatureChange);

            //run any events in the message queue - otherwise close will hang if there are any outstanding events
            Application.DoEvents();

            tempSensor.close();
        }

        private void thermoTypeComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            try
            {
                switch (thermoTypeComboBox.SelectedItem.ToString())
                {
                    case "K-Type":
                        tempSensor.thermocouples[0].Type = TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_K_TYPE;
                        break;
                    case "J-Type":
                        tempSensor.thermocouples[0].Type = TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_J_TYPE;
                        break;
                    case "E-Type":
                        tempSensor.thermocouples[0].Type = TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_E_TYPE;
                        break;
                    case "T-Type":
                        tempSensor.thermocouples[0].Type = TemperatureSensorSensor.ThermocoupleType.PHIDGET_TEMPERATURE_SENSOR_T_TYPE;
                        break;
                }
                thermoRange.Text = "(" + tempSensor.thermocouples[0].TemperatureMin.ToString() + "C - " +
                    tempSensor.thermocouples[0].TemperatureMax.ToString() + "C)";
            }
            catch (PhidgetException)
            { thermoTypeComboBox.Enabled = false; }
        }

        
    }
}