// - TemperatureSensor simple -
// This simple example simple opens a temperature sensor phidget and waits for one to be attached.  The program will then wait for
// user input to terminate.  While waiting it will display the data generated by the events, such as the temperature change event
// which will display the currently measured temperature without sensitivity modifications.
//
// Copyright 2008 Phidgets Inc.  All rights reserved.
// This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
// view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/


#include <stdio.h>
#include <phidget21.h>

int CCONV AttachHandler(CPhidgetHandle TEMP, void *userptr)
{
	int serialNo;
	const char *name;

	CPhidget_getDeviceName (TEMP, &name);
	CPhidget_getSerialNumber(TEMP, &serialNo);
	printf("%s %10d attached!\n", name, serialNo);

	return 0;
}

int CCONV DetachHandler(CPhidgetHandle TEMP, void *userptr)
{
	int serialNo;
	const char *name;

	CPhidget_getDeviceName (TEMP, &name);
	CPhidget_getSerialNumber(TEMP, &serialNo);
	printf("%s %10d detached!\n", name, serialNo);

	return 0;
}

int CCONV ErrorHandler(CPhidgetHandle TEMP, void *userptr, int ErrorCode, const char *Description)
{
	printf("Error handled. %d - %s\n", ErrorCode, Description);
	return 0;
}

int CCONV TemperatureChangeHandler(CPhidgetTemperatureSensorHandle TEMP, void *usrptr, int Index, double Value)
{
	double ambient;
	CPhidgetTemperatureSensor_getAmbientTemperature(TEMP, &ambient);
	printf("Temperature sensor: %d > Temperature: %f (Ambient: %f)\n", Index, Value, ambient);
	return 0;
}

//Display the properties of the attached phidget to the screen.  We will be displaying the name, serial number and version of the attached device.
int display_properties(CPhidgetTemperatureSensorHandle phid)
{
	int serialNo, version, numInputs, i;
	double value;
	const char* ptr;

	double min, max;

	CPhidget_getDeviceType((CPhidgetHandle)phid, &ptr);
	CPhidget_getSerialNumber((CPhidgetHandle)phid, &serialNo);
	CPhidget_getDeviceVersion((CPhidgetHandle)phid, &version);

	CPhidgetTemperatureSensor_getTemperatureInputCount (phid, &numInputs);
	

	printf("%s\n", ptr);
	printf("Serial Number: %10d\nVersion: %8d\n", serialNo, version);
	printf("# Temperature Inputs: %d\n", numInputs);

	for(i = 0; i < numInputs; i++)
	{
		CPhidgetTemperatureSensor_getTemperatureChangeTrigger (phid, i, &value);
		CPhidgetTemperatureSensor_getTemperatureMax(phid, i, &max);
		CPhidgetTemperatureSensor_getTemperatureMin(phid, i, &min);
		printf("Temperature Input #: %d > sensitivity: %f Max: %0.0f, Min: %0.0f\n", i, value, max, min);
	}

	CPhidgetTemperatureSensor_getPotentialMax(phid, 0, &max);
	CPhidgetTemperatureSensor_getPotentialMin(phid, 0, &min);
	printf("Potential Max: %0.3f, Min: %0.3f\n", max, min);

	CPhidgetTemperatureSensor_getAmbientTemperatureMax(phid, &max);
	CPhidgetTemperatureSensor_getAmbientTemperatureMin(phid, &min);
	printf("Ambient Sensor Max: %0.0f, Min: %0.0f\n", max, min);

	return 0;
}

int tempsensor_simple()
{
	int result;
	const char *err;

	//Declare an temperature sensor handle
	CPhidgetTemperatureSensorHandle temp = 0;

	//create the temperature sensor object
	CPhidgetTemperatureSensor_create(&temp);

	//Set the handlers to be run when the device is plugged in or opened from software, unplugged or closed from software, or generates an error.
	CPhidget_set_OnAttach_Handler((CPhidgetHandle)temp, AttachHandler, NULL);
	CPhidget_set_OnDetach_Handler((CPhidgetHandle)temp, DetachHandler, NULL);
	CPhidget_set_OnError_Handler((CPhidgetHandle)temp, ErrorHandler, NULL);

	//Registers a callback that will run if the Temperature changes by more than the Temperature trigger.
	//Requires the handle for the Temperature Sensor, the function that will be called, and a arbitrary pointer that will be supplied to the callback function (may be NULL).
	CPhidgetTemperatureSensor_set_OnTemperatureChange_Handler(temp, TemperatureChangeHandler, NULL);

	//open the temperature sensor for device connections
	CPhidget_open((CPhidgetHandle)temp, -1);

	//get the program to wait for an temperature sensor device to be attached
	printf("Waiting for TemperatureSensor to be attached....");
	if((result = CPhidget_waitForAttachment((CPhidgetHandle)temp, 10000)))
	{
		CPhidget_getErrorDescription(result, &err);
		printf("Problem waiting for attachment: %s\n", err);
		return 0;
	}

	//Display the properties of the attached accelerometer device
	display_properties(temp);

	//read temperature sensor event data
	printf("Reading.....\n");

	//keep displaying temperature sensor event data until user input is read

	//modify the sensor sensitivity, index 1 is the thermocouple sensor, index 0 is the onboard or ambient sensor
	printf("Setting sensitivity of the thermocouple to 2.00. Press any key to continue\n");
	getchar();

	CPhidgetTemperatureSensor_setTemperatureChangeTrigger (temp, 1, 2.00);

	printf("Press any key to end\n");
	getchar();

	//since user input has been read, this is a signal to terminate the program so we will close the phidget and delete the object we created
	printf("Closing...\n");
	CPhidget_close((CPhidgetHandle)temp);
	CPhidget_delete((CPhidgetHandle)temp);

	//all done, exit
	return 0;
}

int main(int argc, char* argv[])
{
	tempsensor_simple();
	return 0;
}

