// - Encoder multiple -
// This example is a very simple implementation of reading data from two attached PhidgetEncoders.
// This example is meant to give you a very general idea of how to manage opening two encoders simultaneously.
// Your specific implementation might require a bit more tweaking.  The process laid out in this example
// also applies to pretty much every other type of phidget when opening multiple of the same type of phidget.
// IMPORTANT:  Remember to check the serial numbers for the encoders you are using and to change the values being used
// in the CPhidget_open calls.
//
// Copyright 2008 Phidgets Inc.  All rights reserved.
// This work is licensed under the Creative Commons Attribution 2.5 Canada License. 
// view a copy of this license, visit http://creativecommons.org/licenses/by/2.5/ca/

#include "stdafx.h"
#include "..\..\..\phidget21.h"

int __stdcall AttachHandler(CPhidgetHandle ENC, void *userptr)
{
	int serialNo;
	CPhidget_getSerialNumber(ENC, &serialNo);
	printf("Encoder %10d attached! \n", serialNo);

	return 0;
}

int __stdcall DetachHandler(CPhidgetHandle ENC, void *userptr)
{
	int serialNo;
	CPhidget_getSerialNumber(ENC, &serialNo);
	printf("Encoder %10d detached! \n", serialNo);

	return 0;
}

int __stdcall ErrorHandler(CPhidgetHandle ENC, void *userptr, int ErrorCode, const char *Description)
{
	int serialNo;
	CPhidget_getSerialNumber(ENC, &serialNo);
	printf("Error handled. Encoder %10d: %i - %s \n", serialNo, ErrorCode, Description);

	return 0;
}

int __stdcall InputChangeHandler(CPhidgetEncoderHandle ENC, void *usrptr, int Index, int State)
{
	int serialNo;
	CPhidget_getSerialNumber((CPhidgetHandle)ENC, &serialNo);
	printf("Encoder %10d: Input #%i - State: %i \n", serialNo, Index, State);

	return 0;
}

int __stdcall PositionChangeHandler(CPhidgetEncoderHandle ENC, void *usrptr, int Index, int Time, int RelativePosition)
{
	int serialNo;
	CPhidget_getSerialNumber((CPhidgetHandle)ENC, &serialNo);
	int Position;
	CPhidgetEncoder_getPosition(ENC, Index, &Position);
	printf("Encoder %10d: Index #%i - Position: %5d -- Relative Change %2d -- Elapsed Time: %5d \n", serialNo, Index, Position, RelativePosition, Time);
	
	return 0;
}

//Display the properties of the attached phidget to the screen.  We will be displaying the name, serial number and version of the attached device.
//Will also display the number of inputs and encoders on this device
int display_properties(CPhidgetEncoderHandle phid)
{
	int serialNo, version, num_inputs, num_encoders;
	const char* ptr;

	CPhidget_getDeviceType((CPhidgetHandle)phid, &ptr);
	CPhidget_getSerialNumber((CPhidgetHandle)phid, &serialNo);
	CPhidget_getDeviceVersion((CPhidgetHandle)phid, &version);

	CPhidgetEncoder_getInputCount(phid, &num_inputs);
	CPhidgetEncoder_getEncoderCount(phid, &num_encoders);

	printf("%s\n", ptr);
	printf("Serial Number: %10d\nVersion: %8d\n", serialNo, version);
	printf("Num Encoders: %d\nNum Inputs: %d\n", num_encoders, num_inputs);

	return 0;
}

int encoder_multiple()
{
	int result;
	const char *err;

	//Declare an encoder handle for each of the two encoders we will be using
	CPhidgetEncoderHandle encoder1 = 0;
	CPhidgetEncoderHandle encoder2 = 0;

	//create the encoder object
	CPhidgetEncoder_create(&encoder1);
	CPhidgetEncoder_create(&encoder2);

	//Set the handlers to be run when the device is plugged in or opened from software, unplugged or closed from software, or generates an error.
	CPhidget_set_OnAttach_Handler((CPhidgetHandle)encoder1, AttachHandler, NULL);
	CPhidget_set_OnDetach_Handler((CPhidgetHandle)encoder1, DetachHandler, NULL);
	CPhidget_set_OnError_Handler((CPhidgetHandle)encoder1, ErrorHandler, NULL);
	CPhidget_set_OnAttach_Handler((CPhidgetHandle)encoder2, AttachHandler, NULL);
	CPhidget_set_OnDetach_Handler((CPhidgetHandle)encoder2, DetachHandler, NULL);
	CPhidget_set_OnError_Handler((CPhidgetHandle)encoder2, ErrorHandler, NULL);

	//Registers a callback that will run if an input changes.
	//Requires the handle for the Phidget, the function that will be called, and an arbitrary pointer that will be supplied to the callback function (may be NULL).
	CPhidgetEncoder_set_OnInputChange_Handler(encoder1, InputChangeHandler, NULL);
	CPhidgetEncoder_set_OnInputChange_Handler(encoder2, InputChangeHandler, NULL);

	//Registers a callback that will run if the encoder changes.
	//Requires the handle for the Encoder, the function that will be called, and an arbitrary pointer that will be supplied to the callback function (may be NULL).
	CPhidgetEncoder_set_OnPositionChange_Handler (encoder1, PositionChangeHandler, NULL);
	CPhidgetEncoder_set_OnPositionChange_Handler (encoder2, PositionChangeHandler, NULL);

	//open the two encoders....need to specifiy the serial number for each encoder to open that encoder.
	CPhidget_open((CPhidgetHandle)encoder1, 29577);

	//get the program to wait for an accelerometer device to be attached
	printf("Waiting for Encoder 1 (29577) to be attached....\n");
	if(result = CPhidget_waitForAttachment((CPhidgetHandle)encoder1, 10000))
	{
		CPhidget_getErrorDescription(result, &err);
		printf("Problem waiting for attachment: %s\n", err);
		return 0;
	}

	//Display the properties of the attached encoder device
	display_properties(encoder1);

	CPhidget_open((CPhidgetHandle)encoder2, 7019);

	//get the program to wait for an encoder device to be attached
	printf("Waiting for Encoder 2 (7019) to be attached....\n");
	if(result = CPhidget_waitForAttachment((CPhidgetHandle)encoder2, 10000))
	{
		CPhidget_getErrorDescription(result, &err);
		printf("Problem waiting for attachment: %s\n", err);
		return 0;
	}

	//Display the properties of the attached encoder device
	display_properties(encoder2);

	//read encoder event data
	printf("Reading encoder data.....\n");

	//keep displaying encoder data until user input is read
	printf("Press any key to end\n");
	getchar();

	//since user input has been read, this is a signal to terminate the program so we will close the phidget and delete the object we created
	printf("Closing...\n");
	CPhidget_close((CPhidgetHandle)encoder1);
	CPhidget_delete((CPhidgetHandle)encoder1);
	CPhidget_close((CPhidgetHandle)encoder2);
	CPhidget_delete((CPhidgetHandle)encoder2);

	//all done, exit
	return 0;
}

int _tmain(int argc, _TCHAR* argv[])
{
	encoder_multiple();
	return 0;
}

